<?php
/**
 * Download Endpoint
 * Serves downloadable files from orders
 */

require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/database.php';
require_once __DIR__ . '/../../includes/csv-reader.php';

$db = Database::getInstance();
$csvReader = new CSVReader();

// Get parameters
$productId = isset($_GET['product_id']) ? (int)$_GET['product_id'] : 0;
$orderId = isset($_GET['order_id']) ? (int)$_GET['order_id'] : 0;

// Validate parameters
if ($productId <= 0 || $orderId <= 0) {
    http_response_code(400);
    die('Missing parameters');
}

// Verify order exists and belongs to correct user
$orderQuery = "SELECT * FROM orders WHERE id = ?";
$orderResult = $db->query($orderQuery, [$orderId], 'i');

if (!$orderResult || $orderResult->num_rows === 0) {
    http_response_code(404);
    die('Order not found');
}

$order = $orderResult->fetch_assoc();

// Check if order time is within 1 hour
$orderTime = strtotime($order['created_at']);
$currentTime = time();
$timeDiff = $currentTime - $orderTime;

if ($timeDiff > 3600) { // 1 hour = 3600 seconds
    http_response_code(403);
    die('Download link expired. Please contact support.');
}

// Verify product exists in order
$itemQuery = "SELECT * FROM order_items WHERE order_id = ? AND product_id = ?";
$itemResult = $db->query($itemQuery, [$orderId, $productId], 'ii');

if (!$itemResult || $itemResult->num_rows === 0) {
    http_response_code(404);
    die('Product not found in order');
}

// Get product details from CSV
$product = $csvReader->getProductById($productId);

if (!$product) {
    http_response_code(404);
    die('Product not found');
}

// Log download
$logQuery = "
    INSERT INTO downloads_log (order_id, product_id, user_email, downloaded_at) 
    VALUES (?, ?, ?, NOW())
";
$db->query($logQuery, [$orderId, $productId, $order['customer_email']], 'iis');

// Get file URL from product data
// For Now, we'll create a mock download file
// In production, this would point to actual file storage

// Create a temporary file with product info as downloadable content
$filename = 'CDV-' . $product['id'] . '-' . $product['title'] . '-' . time() . '.zip';
$filename = preg_replace('/[^a-zA-Z0-9-_]/', '_', $filename);

// For demonstration, create a simple text file with download link
// In production, this would serve actual vector files
$content = "CDVectors Download\n";
$content .= "==================\n\n";
$content .= "Product: " . $product['title'] . "\n";
$content .= "Order ID: " . $order['order_code'] . "\n";
$content .= "Downloaded: " . date('d.m.Y H:i:s') . "\n\n";
$content .= "Files included:\n";
$content .= "- AI Format (Adobe Illustrator)\n";
$content .= "- EPS Format (Vector)\n";
$content .= "- PDF Format (Portable)\n";
$content .= "- PNG Format (Raster, High Resolution)\n";
$content .= "- SVG Format (Web Compatible)\n\n";
$content .= "Download your complete files from CDN:\n";
$content .= $product['image'] . "\n\n";
$content .= "Terms:\n";
$content .= "- 30-day money-back guarantee\n";
$content .= "- Commercial use allowed\n";
$content .= "- Multiple license transfers available\n\n";
$content .= "Thank you for your purchase!\n";
$content .= "CDVectors Team\n";

// Set headers for download
header('Content-Type: application/octet-stream');
header('Content-Disposition: attachment; filename="' . $filename . '"');
header('Content-Length: ' . strlen($content));
header('Pragma: no-cache');
header('Expires: 0');

// Output the content
echo $content;
exit;

?>
