<?php
/**
 * Order Creation API Endpoint
 * Creates a new order in the database
 */

require_once __DIR__ . '/../../includes/config.php';
require_once __DIR__ . '/../../includes/database.php';

header('Content-Type: application/json');

// Only allow POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Get JSON data
$data = json_decode(file_get_contents('php://input'), true);

if (!$data || !isset($data['customer']) || !isset($data['cart']) || !isset($data['total'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Missing required data']);
    exit();
}

$db = Database::getInstance();
$customer = $data['customer'];
$cart = $data['cart'];
$total = (float)$data['total'];
$userId = isset($data['user_id']) ? (int)$data['user_id'] : null;

// Validate cart
if (empty($cart)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Cart is empty']);
    exit();
}

// Generate unique order code: CDV-YYYY-NNNNN
$year = date('Y');
$orderCode = 'CDV-' . $year . '-' . str_pad(rand(1, 99999), 5, '0', STR_PAD_LEFT);

// Start transaction
$db->beginTransaction();

try {
    // Insert order into database
    $orderQuery = "
        INSERT INTO orders (
            order_code, 
            user_id, 
            customer_name, 
            customer_email, 
            customer_phone,
            billing_address, 
            billing_city, 
            billing_postal,
            total_amount, 
            payment_status, 
            order_status,
            notes,
            created_at
        ) VALUES (
            ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW()
        )
    ";

    $stmt = $db->query(
        $orderQuery,
        [
            $orderCode,
            $userId,
            $customer['name'],
            $customer['email'],
            $customer['phone'],
            $customer['address'],
            $customer['city'],
            $customer['postal'],
            $total,
            'pending', // payment_status
            'pending', // order_status
            $customer['notes'] ?? ''
        ],
        'ssissssss'
    );

    // Get the inserted order ID
    $orderId = $db->getInsertId();

    if (!$orderId) {
        throw new Exception('Failed to create order');
    }

    // Insert order items
    foreach ($cart as $item) {
        $itemQuery = "
            INSERT INTO order_items (
                order_id,
                product_id,
                quantity
            ) VALUES (?, ?, ?)
        ";

        $db->query(
            $itemQuery,
            [$orderId, $item['id'], $item['quantity']],
            'iii'
        );
    }

    // Commit transaction
    $db->commit();

    // Return success response
    http_response_code(201);
    echo json_encode([
        'success' => true,
        'message' => 'Order created successfully',
        'order_id' => $orderId,
        'order_code' => $orderCode
    ]);

} catch (Exception $e) {
    // Rollback on error
    $db->rollback();

    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error creating order: ' . $e->getMessage()
    ]);
}
?>
