<?php
/**
 * User Account Dashboard
 * Shows user orders, downloads, and account information
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/auth.php';
require_once __DIR__ . '/includes/database.php';

$auth = new Auth();

// Check if user is logged in
if (!$auth->isLoggedIn()) {
    header('Location: ' . APP_URL . '/login.php');
    exit();
}

$db = Database::getInstance();
$pageTitle = 'Hesabım';
$user = $_SESSION['user'];

// Get user's orders
$ordersQuery = "
    SELECT * FROM orders 
    WHERE user_id = ? OR customer_email = ?
    ORDER BY created_at DESC
    LIMIT 10
";
$ordersResult = $db->query($ordersQuery, [$user['id'], $user['email']], 'iss');
$orders = [];
if ($ordersResult) {
    while ($row = $ordersResult->fetch_assoc()) {
        $orders[] = $row;
    }
}

// Get user's total spending
$spendingQuery = "
    SELECT 
        COUNT(*) as total_orders,
        COALESCE(SUM(total_amount), 0) as total_spent
    FROM orders 
    WHERE user_id = ? OR customer_email = ?
";
$spendingResult = $db->query($spendingQuery, [$user['id'], $user['email']], 'iss');
$spending = ['total_orders' => 0, 'total_spent' => 0];
if ($spendingResult && $row = $spendingResult->fetch_assoc()) {
    $spending = $row;
}

require_once __DIR__ . '/includes/header.php';
?>

<div style="max-width: 1200px; margin: 0 auto; padding: 2rem 0;">
    <!-- Welcome Banner -->
    <div style="
        background: linear-gradient(135deg, #ff6b9d 0%, #e63946 100%);
        color: white;
        padding: 3rem 2rem;
        border-radius: 12px;
        margin-bottom: 2rem;
    ">
        <h1 style="margin: 0; font-size: 2.5rem;">👋 Hoş Geldiniz, <?php echo htmlspecialchars($user['username']); ?>!</h1>
        <p style="margin: 0.5rem 0 0 0; opacity: 0.95; font-size: 1.1rem;">
            Hesap yönetim paneline hoşgeldiniz. Sipariş ve dosyalarınızı buradan kontrol edebilirsiniz.
        </p>
    </div>

    <!-- Dashboard Grid -->
    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 1.5rem; margin-bottom: 3rem;">
        <!-- Total Orders -->
        <div style="
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.08);
            text-align: center;
        ">
            <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">📦</div>
            <h3 style="margin: 0.5rem 0; color: #666;">Toplam Siparişler</h3>
            <p style="margin: 0; font-size: 2rem; font-weight: 600; color: #e63946;">
                <?php echo $spending['total_orders']; ?>
            </p>
        </div>

        <!-- Total Spent -->
        <div style="
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.08);
            text-align: center;
        ">
            <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">💰</div>
            <h3 style="margin: 0.5rem 0; color: #666;">Toplam Harcama</h3>
            <p style="margin: 0; font-size: 2rem; font-weight: 600; color: #e63946;">
                $<?php echo number_format($spending['total_spent'], 2); ?>
            </p>
        </div>

        <!-- Account Status -->
        <div style="
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.08);
            text-align: center;
        ">
            <div style="font-size: 2.5rem; margin-bottom: 0.5rem;">✅</div>
            <h3 style="margin: 0.5rem 0; color: #666;">Hesap Durumu</h3>
            <p style="margin: 0; font-size: 1.1rem; font-weight: 600;">
                <span style="
                    background: #d4edda;
                    color: #155724;
                    padding: 0.5rem 1rem;
                    border-radius: 20px;
                    display: inline-block;
                ">
                    Aktif
                </span>
            </p>
        </div>
    </div>

    <!-- Main Content Grid -->
    <div style="display: grid; grid-template-columns: 2fr 1fr; gap: 2rem;">
        <!-- Recent Orders Section -->
        <div>
            <h2 style="margin-bottom: 1.5rem;">📋 Son Siparişlerim</h2>
            
            <?php if (!empty($orders)): ?>
                <div style="
                    background: white;
                    border-radius: 12px;
                    overflow: hidden;
                    box-shadow: 0 2px 12px rgba(0,0,0,0.08);
                ">
                    <table style="width: 100%; border-collapse: collapse;">
                        <thead>
                            <tr style="background: #faf8f6; border-bottom: 2px solid #eee;">
                                <th style="padding: 1rem; text-align: left; font-weight: 600;">Sipariş No</th>
                                <th style="padding: 1rem; text-align: left; font-weight: 600;">Tarih</th>
                                <th style="padding: 1rem; text-align: left; font-weight: 600;">Tutar</th>
                                <th style="padding: 1rem; text-align: left; font-weight: 600;">Durum</th>
                                <th style="padding: 1rem; text-align: center; font-weight: 600;">İşlem</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($orders as $order): ?>
                                <tr style="border-bottom: 1px solid #eee;">
                                    <td style="padding: 1rem; color: #e63946; font-weight: 600;">
                                        <?php echo htmlspecialchars($order['order_code']); ?>
                                    </td>
                                    <td style="padding: 1rem;">
                                        <?php echo date('d.m.Y', strtotime($order['created_at'])); ?>
                                    </td>
                                    <td style="padding: 1rem; font-weight: 600;">
                                        $<?php echo number_format($order['total_amount'], 2); ?>
                                    </td>
                                    <td style="padding: 1rem;">
                                        <span style="
                                            padding: 0.4rem 0.8rem;
                                            border-radius: 6px;
                                            font-size: 0.85rem;
                                            font-weight: 600;
                                            background: <?php 
                                                echo $order['payment_status'] === 'paid' ? '#d4edda' : '#fff3cd';
                                            ?>;
                                            color: <?php 
                                                echo $order['payment_status'] === 'paid' ? '#155724' : '#997404';
                                            ?>;
                                        ">
                                            <?php 
                                                echo $order['payment_status'] === 'paid' ? '✅ Ödendi' : '⏳ Beklemede';
                                            ?>
                                        </span>
                                    </td>
                                    <td style="padding: 1rem; text-align: center;">
                                        <a 
                                            href="<?php echo APP_URL; ?>/my-downloads.php?order_id=<?php echo $order['id']; ?>"
                                            style="
                                                color: #e63946;
                                                text-decoration: none;
                                                font-weight: 600;
                                                padding: 0.5rem 1rem;
                                                border-radius: 6px;
                                                transition: all 0.3s ease;
                                            "
                                            onmouseover="this.style.backgroundColor='#ffe6e6';"
                                            onmouseout="this.style.backgroundColor='transparent';"
                                        >
                                            İndir →
                                        </a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php else: ?>
                <div style="
                    background: white;
                    padding: 3rem 2rem;
                    border-radius: 12px;
                    text-align: center;
                    box-shadow: 0 2px 12px rgba(0,0,0,0.08);
                ">
                    <p style="color: #999; margin: 0;">Henüz hiçbir sipariş vermemişsiniz.</p>
                    <a 
                        href="<?php echo APP_URL; ?>/products.php"
                        style="
                            display: inline-block;
                            margin-top: 1rem;
                            background: linear-gradient(135deg, #ff6b9d 0%, #e63946 100%);
                            color: white;
                            padding: 0.75rem 1.5rem;
                            border-radius: 8px;
                            text-decoration: none;
                            font-weight: 600;
                        "
                    >
                        Ürünlere Göz At 💕
                    </a>
                </div>
            <?php endif; ?>
        </div>

        <!-- Sidebar -->
        <div>
            <!-- Account Information -->
            <div style="
                background: white;
                padding: 2rem;
                border-radius: 12px;
                box-shadow: 0 2px 12px rgba(0,0,0,0.08);
                margin-bottom: 2rem;
            ">
                <h3 style="margin-top: 0;">👤 Hesap Bilgileri</h3>
                <div style="border-top: 1px solid #eee; padding-top: 1rem;">
                    <p style="margin: 0.5rem 0; color: #999;">
                        <strong>Kullanıcı Adı:</strong>
                    </p>
                    <p style="margin: 0 0 1rem 0;">
                        <?php echo htmlspecialchars($user['username']); ?>
                    </p>
                    
                    <p style="margin: 0.5rem 0; color: #999;">
                        <strong>Email:</strong>
                    </p>
                    <p style="margin: 0 0 1rem 0;">
                        <?php echo htmlspecialchars($user['email']); ?>
                    </p>
                    
                    <p style="margin: 0.5rem 0; color: #999;">
                        <strong>Üyelik Tarihi:</strong>
                    </p>
                    <p style="margin: 0;">
                        <?php echo isset($user['created_at']) ? date('d.m.Y', strtotime($user['created_at'])) : 'Belirtilmemiş'; ?>
                    </p>
                </div>
            </div>

            <!-- Quick Actions -->
            <div style="
                background: #faf8f6;
                padding: 2rem;
                border-radius: 12px;
                border: 2px solid #eee;
            ">
                <h3 style="margin-top: 0;">⚙️ İşlemler</h3>
                <div style="display: flex; flex-direction: column; gap: 1rem;">
                    <a 
                        href="<?php echo APP_URL; ?>/account-settings.php"
                        style="
                            display: block;
                            padding: 0.75rem 1rem;
                            background: white;
                            color: #e63946;
                            text-decoration: none;
                            border: 2px solid #ff6b9d;
                            border-radius: 8px;
                            text-align: center;
                            font-weight: 600;
                            transition: all 0.3s ease;
                        "
                        onmouseover="this.style.backgroundColor='#ffe6e6';"
                        onmouseout="this.style.backgroundColor='white';"
                    >
                        Ayarları Değiştir
                    </a>
                    
                    <a 
                        href="<?php echo APP_URL; ?>/change-password.php"
                        style="
                            display: block;
                            padding: 0.75rem 1rem;
                            background: white;
                            color: #e63946;
                            text-decoration: none;
                            border: 2px solid #ff6b9d;
                            border-radius: 8px;
                            text-align: center;
                            font-weight: 600;
                            transition: all 0.3s ease;
                        "
                        onmouseover="this.style.backgroundColor='#ffe6e6';"
                        onmouseout="this.style.backgroundColor='white';"
                    >
                        Şifre Değiştir
                    </a>
                    
                    <a 
                        href="<?php echo APP_URL; ?>/logout.php"
                        style="
                            display: block;
                            padding: 0.75rem 1rem;
                            background: #e63946;
                            color: white;
                            text-decoration: none;
                            border-radius: 8px;
                            text-align: center;
                            font-weight: 600;
                            transition: all 0.3s ease;
                        "
                        onmouseover="this.style.opacity='0.9';"
                        onmouseout="this.style.opacity='1';"
                    >
                        Çıkış Yap
                    </a>
                </div>
            </div>

            <!-- Support -->
            <div style="
                background: #f0f8ff;
                padding: 2rem;
                border-radius: 12px;
                border: 2px solid #87ceeb;
                margin-top: 2rem;
            ">
                <h4 style="margin-top: 0;">❓ Yardım Gerekli Mi?</h4>
                <p style="color: #666; margin: 0 0 1rem 0; font-size: 0.95rem;">
                    Bir sorun mu var? Destek ekibimiz size yardımcı olmak için hazır.
                </p>
                <a 
                    href="<?php echo APP_URL; ?>/support.php"
                    style="
                        display: block;
                        padding: 0.75rem 1rem;
                        background: #87ceeb;
                        color: white;
                        text-decoration: none;
                        border-radius: 8px;
                        text-align: center;
                        font-weight: 600;
                        transition: all 0.3s ease;
                    "
                    onmouseover="this.style.backgroundColor='#6bb8d6';"
                    onmouseout="this.style.backgroundColor='#87ceeb';"
                >
                    Destek Al
                </a>
            </div>
        </div>
    </div>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
