-- CDVectors Database Schema
-- UTF8MB4 Support for International Characters

SET NAMES utf8mb4;
SET CHARACTER SET utf8mb4;

-- ============================================
-- MODULE 1: USER SYSTEM
-- ============================================

CREATE TABLE IF NOT EXISTS `users` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `username` VARCHAR(50) UNIQUE NOT NULL COMMENT 'Unique username',
    `email` VARCHAR(100) UNIQUE NOT NULL COMMENT 'Unique email address',
    `password` VARCHAR(255) NOT NULL COMMENT 'Bcrypt hashed password',
    `first_name` VARCHAR(50) DEFAULT NULL,
    `last_name` VARCHAR(50) DEFAULT NULL,
    `phone` VARCHAR(20) DEFAULT NULL,
    `country` VARCHAR(50) DEFAULT NULL,
    `city` VARCHAR(50) DEFAULT NULL,
    `address` TEXT DEFAULT NULL,
    `is_admin` BOOLEAN DEFAULT FALSE COMMENT 'Admin access flag',
    `email_verified` BOOLEAN DEFAULT FALSE COMMENT 'Email verification status',
    `verification_token` VARCHAR(255) DEFAULT NULL COMMENT 'Email verification token',
    `account_status` ENUM('active', 'inactive', 'suspended', 'deleted') DEFAULT 'active',
    `total_orders` INT DEFAULT 0 COMMENT 'Total number of orders',
    `total_spent` DECIMAL(10, 2) DEFAULT 0.00 COMMENT 'Total amount spent',
    `last_login` DATETIME DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX `idx_email` (`email`),
    INDEX `idx_username` (`username`),
    INDEX `idx_account_status` (`account_status`),
    INDEX `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='User accounts table';

-- ============================================
-- MODULE 3: PRODUCTS SYSTEM
-- ============================================

CREATE TABLE IF NOT EXISTS `products` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `product_code` VARCHAR(50) UNIQUE NOT NULL COMMENT 'Unique product code (VEK001, VEK002, etc)',
    `title` VARCHAR(255) NOT NULL,
    `description` LONGTEXT,
    `price` DECIMAL(10, 2) NOT NULL,
    `category` VARCHAR(50) NOT NULL COMMENT 'Main category (Wedding, Birthday, etc)',
    `subcategory` VARCHAR(50) DEFAULT NULL COMMENT 'Subcategory (Frames, Borders, etc)',
    `tags` VARCHAR(255) DEFAULT NULL COMMENT 'Comma-separated tags for filtering',
    `image_url` VARCHAR(255) DEFAULT NULL COMMENT 'Main product image',
    `thumbnail_url` VARCHAR(255) DEFAULT NULL COMMENT 'Small thumbnail image',
    `file_url` VARCHAR(255) DEFAULT NULL COMMENT 'Download file path',
    `file_size` INT DEFAULT 0 COMMENT 'File size in bytes',
    `file_type` VARCHAR(20) DEFAULT NULL COMMENT 'File type (zip, ai, eps, svg, etc)',
    `status` ENUM('active', 'inactive', 'draft') DEFAULT 'active' COMMENT 'Product visibility status',
    `stock_status` ENUM('in_stock', 'out_of_stock') DEFAULT 'in_stock',
    `views` INT DEFAULT 0 COMMENT 'Product view counter',
    `downloads` INT DEFAULT 0 COMMENT 'Product download counter',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX `idx_product_code` (`product_code`),
    INDEX `idx_category` (`category`),
    INDEX `idx_status` (`status`),
    INDEX `idx_price` (`price`),
    FULLTEXT INDEX `idx_ft_title_description` (`title`, `description`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Products catalog';

-- ============================================
-- MODULE 4: SHOPPING CART
-- ============================================

CREATE TABLE IF NOT EXISTS `cart` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED DEFAULT NULL COMMENT 'NULL for guest users',
    `session_id` VARCHAR(100) DEFAULT NULL COMMENT 'Session ID for guest users',
    `product_id` INT UNSIGNED NOT NULL,
    `quantity` INT DEFAULT 1,
    `added_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`product_id`) REFERENCES `products`(`id`) ON DELETE CASCADE,
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_session_id` (`session_id`),
    UNIQUE KEY `unique_user_product` (`user_id`, `product_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Shopping cart items';

-- ============================================
-- MODULE 5: ORDERS
-- ============================================

CREATE TABLE IF NOT EXISTS `orders` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `order_code` VARCHAR(50) UNIQUE NOT NULL COMMENT 'CDV-YYYY-NNN format',
    `user_id` INT UNSIGNED DEFAULT NULL,
    `paddle_order_id` VARCHAR(100) DEFAULT NULL COMMENT 'Paddle order reference',
    `paddle_transaction_id` VARCHAR(100) DEFAULT NULL COMMENT 'Paddle transaction reference',
    `total_amount` DECIMAL(10, 2) NOT NULL,
    `currency` VARCHAR(3) DEFAULT 'USD',
    `status` ENUM('pending', 'paid', 'processing', 'completed', 'cancelled', 'refunded') DEFAULT 'pending',
    `payment_status` ENUM('unpaid', 'paid', 'refunded', 'chargeback') DEFAULT 'unpaid',
    `order_status` ENUM('pending', 'processing', 'completed', 'shipped', 'cancelled') DEFAULT 'pending',
    `customer_email` VARCHAR(100) NOT NULL,
    `customer_name` VARCHAR(100) NOT NULL,
    `customer_phone` VARCHAR(20) DEFAULT NULL,
    `billing_address` TEXT DEFAULT NULL,
    `billing_city` VARCHAR(50) DEFAULT NULL,
    `billing_postal` VARCHAR(20) DEFAULT NULL,
    `customer_ip` VARCHAR(45) DEFAULT NULL,
    `refund_requested` BOOLEAN DEFAULT FALSE,
    `refund_reason` TEXT DEFAULT NULL,
    `refund_status` ENUM('none', 'pending', 'approved', 'rejected', 'processed') DEFAULT 'none',
    `downloaded` BOOLEAN DEFAULT FALSE COMMENT 'Whether customer has downloaded files',
    `notes` TEXT DEFAULT NULL COMMENT 'Order notes from customer',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `paid_at` DATETIME DEFAULT NULL,
    `completed_at` DATETIME DEFAULT NULL,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL,
    INDEX `idx_order_code` (`order_code`),
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_customer_email` (`customer_email`),
    INDEX `idx_status` (`status`),
    INDEX `idx_created_at` (`created_at`),
    INDEX `idx_paddle_order_id` (`paddle_order_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Customer orders';

CREATE TABLE IF NOT EXISTS `order_items` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `order_id` INT UNSIGNED NOT NULL,
    `product_id` INT UNSIGNED NOT NULL,
    `product_code` VARCHAR(50) NOT NULL,
    `product_title` VARCHAR(255) NOT NULL,
    `quantity` INT DEFAULT 1,
    `price` DECIMAL(10, 2) NOT NULL COMMENT 'Price at time of purchase',
    `subtotal` DECIMAL(10, 2) NOT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (`order_id`) REFERENCES `orders`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`product_id`) REFERENCES `products`(`id`) ON DELETE RESTRICT,
    INDEX `idx_order_id` (`order_id`),
    INDEX `idx_product_id` (`product_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Items in each order';

-- ============================================
-- MODULE 8: REFUNDS
-- ============================================

CREATE TABLE IF NOT EXISTS `refund_requests` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `order_id` INT UNSIGNED NOT NULL,
    `user_id` INT UNSIGNED DEFAULT NULL,
    `refund_amount` DECIMAL(10, 2) NOT NULL,
    `reason` TEXT NOT NULL,
    `status` ENUM('pending', 'approved', 'rejected', 'processed') DEFAULT 'pending',
    `admin_notes` TEXT DEFAULT NULL,
    `processed_at` DATETIME DEFAULT NULL,
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    FOREIGN KEY (`order_id`) REFERENCES `orders`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE SET NULL,
    INDEX `idx_order_id` (`order_id`),
    INDEX `idx_status` (`status`),
    INDEX `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Refund requests';

-- ============================================
-- MODULE 9: SUPPORT TICKETS
-- ============================================

CREATE TABLE IF NOT EXISTS `support_tickets` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `ticket_number` VARCHAR(50) UNIQUE NOT NULL COMMENT 'Ticket reference number',
    `order_id` INT UNSIGNED DEFAULT NULL,
    `user_id` INT UNSIGNED NOT NULL,
    `subject` VARCHAR(255) NOT NULL,
    `message` LONGTEXT NOT NULL,
    `status` ENUM('open', 'in_progress', 'resolved', 'closed') DEFAULT 'open',
    `ticket_type` ENUM('refund', 'technical', 'general', 'file_issue') DEFAULT 'general',
    `admin_response` LONGTEXT DEFAULT NULL,
    `admin_id` INT UNSIGNED DEFAULT NULL COMMENT 'Admin who responded',
    `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    `updated_at` DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    `resolved_at` DATETIME DEFAULT NULL,
    
    FOREIGN KEY (`order_id`) REFERENCES `orders`(`id`) ON DELETE SET NULL,
    FOREIGN KEY (`user_id`) REFERENCES `users`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`admin_id`) REFERENCES `users`(`id`) ON DELETE SET NULL,
    INDEX `idx_ticket_number` (`ticket_number`),
    INDEX `idx_user_id` (`user_id`),
    INDEX `idx_status` (`status`),
    INDEX `idx_created_at` (`created_at`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Support tickets';

-- ============================================
-- MODULE 10: DOWNLOADS LOG
-- ============================================

CREATE TABLE IF NOT EXISTS `downloads_log` (
    `id` INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    `order_id` INT UNSIGNED NOT NULL,
    `product_id` INT UNSIGNED NOT NULL,
    `user_email` VARCHAR(100) NOT NULL,
    `downloaded_at` DATETIME DEFAULT CURRENT_TIMESTAMP,
    
    FOREIGN KEY (`order_id`) REFERENCES `orders`(`id`) ON DELETE CASCADE,
    FOREIGN KEY (`product_id`) REFERENCES `products`(`id`) ON DELETE CASCADE,
    INDEX `idx_order_id` (`order_id`),
    INDEX `idx_product_id` (`product_id`),
    INDEX `idx_downloaded_at` (`downloaded_at`),
    INDEX `idx_user_email` (`user_email`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Download history log';

-- ============================================
-- END OF SCHEMA
-- ============================================
