<?php
/**
 * CDVectors Authentication Class
 * =============================
 * Handles user registration, login, session management
 */

require_once __DIR__ . '/database.php';

class Auth {
    private Database $db;

    public function __construct() {
        $this->db = Database::getInstance();
    }

    /**
     * Register new user
     * @return array ['success' => bool, 'message' => string, 'user_id' => int|null]
     */
    public function register(array $data): array {
        // Validate input
        $validation = $this->validateRegisterInput($data);
        if (!$validation['valid']) {
            return ['success' => false, 'message' => $validation['errors']];
        }

        // Extract data
        $username = trim($data['username']);
        $email = trim(strtolower($data['email']));
        $password = $data['password'];
        $firstName = trim($data['first_name'] ?? '');
        $lastName = trim($data['last_name'] ?? '');

        try {
            // Check if user already exists
            if ($this->userExists($username, $email)) {
                return ['success' => false, 'message' => 'Kullanıcı adı veya e-posta zaten kayıtlı.'];
            }

            // Hash password using bcrypt
            $hashedPassword = password_hash($password, PASSWORD_ALGO, PASSWORD_OPTIONS);

            // Generate verification token
            $verificationToken = bin2hex(random_bytes(32));

            // Insert user
            $userId = $this->db->insert('users', [
                'username' => $username,
                'email' => $email,
                'password' => $hashedPassword,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'verification_token' => $verificationToken,
                'account_status' => 'active' // Auto-activate for now, can add email verification later
            ]);

            if (!$userId) {
                return ['success' => false, 'message' => 'Kayıt sırasında bir hata oluştu. Lütfen tekrar deneyin.'];
            }

            // TODO: Send verification email
            // $this->sendVerificationEmail($email, $verificationToken);

            return [
                'success' => true,
                'message' => 'Kayıt başarılı! Şimdi giriş yapabilirsiniz.',
                'user_id' => $userId
            ];
        } catch (Exception $e) {
            error_log('Registration error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Kayıt sırasında bir hata oluştu.'];
        }
    }

    /**
     * Login user
     * @return array ['success' => bool, 'message' => string, 'user_id' => int|null]
     */
    public function login(string $username, string $password): array {
        try {
            // Validate input
            if (empty($username) || empty($password)) {
                return ['success' => false, 'message' => 'Lütfen kullanıcı adı ve şifre giriniz.'];
            }

            // Get user by username or email
            $query = "SELECT * FROM users WHERE (username = ? OR email = ?) AND account_status = 'active'";
            $user = $this->db->getRow($query, [$username, $username]);

            if (!$user) {
                // Security: Don't reveal if username exists
                return ['success' => false, 'message' => 'Kullanıcı adı veya şifre hatalı.'];
            }

            // Verify password
            if (!password_verify($password, $user['password'])) {
                // Optional: Log failed login attempts
                return ['success' => false, 'message' => 'Kullanıcı adı veya şifre hatalı.'];
            }

            // Update last login
            $this->db->update('users', ['last_login' => date('Y-m-d H:i:s')], ['id' => $user['id']]);

            // Set session
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user'] = [
                'id' => $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'first_name' => $user['first_name'],
                'is_admin' => (bool)$user['is_admin']
            ];

            return [
                'success' => true,
                'message' => 'Giriş başarılı!',
                'user_id' => $user['id']
            ];
        } catch (Exception $e) {
            error_log('Login error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Giriş sırasında bir hata oluştu.'];
        }
    }

    /**
     * Logout user
     */
    public function logout(): void {
        // Destroy session
        session_destroy();

        // Redirect to home
        header('Location: ' . APP_URL);
        exit();
    }

    /**
     * Check if user is logged in
     */
    public function isLoggedIn(): bool {
        return isset($_SESSION['user_id']) && !empty($_SESSION['user_id']);
    }

    /**
     * Check if user is admin
     */
    public function isAdmin(): bool {
        return isset($_SESSION['user']) && (bool)$_SESSION['user']['is_admin'];
    }

    /**
     * Require login - redirect if not logged in
     */
    public function requireLogin(): void {
        if (!$this->isLoggedIn()) {
            $_SESSION['redirect_to'] = $_SERVER['REQUEST_URI'];
            header('Location: ' . APP_URL . '/login.php');
            exit();
        }
    }

    /**
     * Require admin - redirect if not admin
     */
    public function requireAdmin(): void {
        if (!$this->isAdmin()) {
            header('HTTP/1.0 403 Forbidden');
            die('Access denied. Admin access required.');
        }
    }

    /**
     * Get current user ID
     */
    public function getCurrentUserId(): ?int {
        return $_SESSION['user_id'] ?? null;
    }

    /**
     * Get current user data
     */
    public function getCurrentUser(): ?array {
        return $_SESSION['user'] ?? null;
    }

    /**
     * Get user by ID
     */
    public function getUserById(int $userId): ?array {
        try {
            return $this->db->getRow("SELECT * FROM users WHERE id = ?", [$userId]);
        } catch (Exception $e) {
            error_log('Get user error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Update user profile
     */
    public function updateProfile(int $userId, array $data): array {
        try {
            // Only allow certain fields to be updated
            $allowedFields = ['first_name', 'last_name', 'phone', 'country', 'city', 'address'];
            $updateData = [];

            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $updateData[$field] = trim($data[$field]);
                }
            }

            if (empty($updateData)) {
                return ['success' => false, 'message' => 'Güncellenecek veri yok.'];
            }

            $result = $this->db->update('users', $updateData, ['id' => $userId]);

            if ($result !== false) {
                return ['success' => true, 'message' => 'Profil başarıyla güncellendi.'];
            } else {
                return ['success' => false, 'message' => 'Profil güncellenirken bir hata oluştu.'];
            }
        } catch (Exception $e) {
            error_log('Update profile error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Bir hata oluştu.'];
        }
    }

    /**
     * Change password
     */
    public function changePassword(int $userId, string $oldPassword, string $newPassword): array {
        try {
            // Validate new password
            $validation = $this->validatePassword($newPassword);
            if (!$validation['valid']) {
                return ['success' => false, 'message' => $validation['errors']];
            }

            // Get user
            $user = $this->getUserById($userId);
            if (!$user) {
                return ['success' => false, 'message' => 'Kullanıcı bulunamadı.'];
            }

            // Verify old password
            if (!password_verify($oldPassword, $user['password'])) {
                return ['success' => false, 'message' => 'Mevcut şifre hatalı.'];
            }

            // Hash new password
            $hashedPassword = password_hash($newPassword, PASSWORD_ALGO, PASSWORD_OPTIONS);

            // Update password
            $this->db->update('users', ['password' => $hashedPassword], ['id' => $userId]);

            return ['success' => true, 'message' => 'Şifre başarıyla değiştirildi.'];
        } catch (Exception $e) {
            error_log('Change password error: ' . $e->getMessage());
            return ['success' => false, 'message' => 'Bir hata oluştu.'];
        }
    }

    /**
     * Check if user exists by username or email
     */
    private function userExists(string $username, string $email): bool {
        try {
            $count = $this->db->count('users', 
                // Using raw query for OR condition
            );
            $query = "SELECT COUNT(*) as count FROM users WHERE username = ? OR email = ?";
            $result = $this->db->query($query, [$username, $email]);
            $row = $result->fetch_assoc();
            return $row['count'] > 0;
        } catch (Exception $e) {
            return true; // Fail safe - assume exists
        }
    }

    /**
     * Validate registration input
     */
    private function validateRegisterInput(array $data): array {
        $errors = [];

        // Username validation
        if (empty($data['username'])) {
            $errors[] = 'Kullanıcı adı gereklidir.';
        } elseif (strlen($data['username']) < 3) {
            $errors[] = 'Kullanıcı adı en az 3 karakter olmalıdır.';
        } elseif (strlen($data['username']) > 50) {
            $errors[] = 'Kullanıcı adı maksimum 50 karakter olmalıdır.';
        } elseif (!preg_match('/^[a-zA-Z0-9_-]+$/', $data['username'])) {
            $errors[] = 'Kullanıcı adı sadece harf, rakam, alt çizgi ve tire içerebilir.';
        }

        // Email validation
        if (empty($data['email'])) {
            $errors[] = 'E-posta gereklidir.';
        } elseif (!filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
            $errors[] = 'Geçerli bir e-posta adresi giriniz.';
        }

        // Password validation
        $passwordValidation = $this->validatePassword($data['password'] ?? '');
        if (!$passwordValidation['valid']) {
            $errors = array_merge($errors, (array)$passwordValidation['errors']);
        }

        // Confirm password
        if (($data['password'] ?? '') !== ($data['confirm_password'] ?? '')) {
            $errors[] = 'Şifreler eşleşmiyor.';
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    /**
     * Validate password strength
     */
    private function validatePassword(string $password): array {
        $errors = [];

        if (empty($password)) {
            $errors[] = 'Şifre gereklidir.';
        } elseif (strlen($password) < 8) {
            $errors[] = 'Şifre en az 8 karakter olmalıdır.';
        } elseif (!preg_match('/[a-z]/', $password)) {
            $errors[] = 'Şifre en az bir küçük harf içermelidir.';
        } elseif (!preg_match('/[A-Z]/', $password)) {
            $errors[] = 'Şifre en az bir büyük harf içermelidir.';
        } elseif (!preg_match('/[0-9]/', $password)) {
            $errors[] = 'Şifre en az bir rakam içermelidir.';
        }

        return [
            'valid' => empty($errors),
            'errors' => $errors
        ];
    }

    /**
     * Generate CSRF token
     */
    public static function generateCsrfToken(): string {
        if (empty($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        }
        return $_SESSION['csrf_token'];
    }

    /**
     * Verify CSRF token
     */
    public static function verifyCsrfToken(string $token): bool {
        return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
    }
}

// ============================================
// END OF AUTH CLASS
// ============================================
