<?php
/**
 * CSV Reader Class
 * ==================
 * Reads and manages products from CSV file
 * Caches data for better performance
 */

class CSVReader {
    private string $csvFilePath;
    private string $cacheDir;
    private string $cacheFile;
    private int $cacheDuration = 3600; // 1 hour

    public function __construct(string $csvFilePath = null) {
        $this->csvFilePath = $csvFilePath ?? __DIR__ . '/../csv/products.csv';
        $this->cacheDir = __DIR__ . '/../cache';
        $this->cacheFile = $this->cacheDir . '/products_cache.json';

        // Create cache directory if not exists
        if (!is_dir($this->cacheDir)) {
            mkdir($this->cacheDir, 0755, true);
        }
    }

    /**
     * Get all products from CSV
     * Uses cache if available
     */
    public function getAllProducts(): array {
        // Check cache first
        if ($this->isCacheValid()) {
            return json_decode(file_get_contents($this->cacheFile), true);
        }

        // Read from CSV
        $products = $this->readCSV();

        // Cache the data
        $this->cacheProducts($products);

        return $products;
    }

    /**
     * Get product by ID
     */
    public function getProductById(int $id): ?array {
        $products = $this->getAllProducts();
        foreach ($products as $product) {
            if ($product['id'] == $id) {
                return $product;
            }
        }
        return null;
    }

    /**
     * Get products by category
     */
    public function getProductsByCategory(string $category): array {
        $products = $this->getAllProducts();
        $result = [];

        foreach ($products as $product) {
            if (strcasecmp($product['category'], $category) === 0) {
                $result[] = $product;
            }
        }

        return $result;
    }

    /**
     * Search products by keyword
     */
    public function searchProducts(string $keyword): array {
        $products = $this->getAllProducts();
        $keyword = strtolower($keyword);
        $result = [];

        foreach ($products as $product) {
            $titleMatch = stripos($product['title'], $keyword) !== false;
            $descMatch = stripos($product['description'], $keyword) !== false;
            $tagsMatch = stripos($product['tags'], $keyword) !== false;

            if ($titleMatch || $descMatch || $tagsMatch) {
                $result[] = $product;
            }
        }

        return $result;
    }

    /**
     * Get popular products (marked as popular in CSV)
     */
    public function getPopularProducts(int $limit = 12): array {
        $products = $this->getAllProducts();
        $popular = [];

        foreach ($products as $product) {
            if ($product['popular'] === 'True' || $product['popular'] === true) {
                $popular[] = $product;
                if (count($popular) >= $limit) break;
            }
        }

        return $popular;
    }

    /**
     * Get categories from products
     */
    public function getCategories(): array {
        $products = $this->getAllProducts();
        $categories = [];

        foreach ($products as $product) {
            $cat = $product['category'];
            if (!in_array($cat, $categories)) {
                $categories[] = $cat;
            }
        }

        sort($categories);
        return $categories;
    }

    /**
     * Get products with pagination
     */
    public function getProductsPaginated(int $page = 1, int $perPage = 12, string $category = null): array {
        if ($category) {
            $products = $this->getProductsByCategory($category);
        } else {
            $products = $this->getAllProducts();
        }

        $totalProducts = count($products);
        $totalPages = ceil($totalProducts / $perPage);

        // Validate page
        $page = max(1, min($page, $totalPages));

        $startIndex = ($page - 1) * $perPage;
        $paginatedProducts = array_slice($products, $startIndex, $perPage);

        return [
            'products' => $paginatedProducts,
            'currentPage' => $page,
            'totalPages' => $totalPages,
            'totalProducts' => $totalProducts,
            'perPage' => $perPage
        ];
    }

    /**
     * Read CSV file
     */
    private function readCSV(): array {
        if (!file_exists($this->csvFilePath)) {
            return [];
        }

        $products = [];
        $handle = fopen($this->csvFilePath, 'r');

        if ($handle === false) {
            return [];
        }

        // Read header
        $header = fgetcsv($handle);
        if (!$header) {
            fclose($handle);
            return [];
        }

        // Read data rows
        while (($row = fgetcsv($handle)) !== false) {
            if (empty($row[0])) continue; // Skip empty rows

            $product = [];
            foreach ($header as $index => $columnName) {
                $value = $row[$index] ?? null;

                // Parse special fields
                if ($columnName === 'tags' || $columnName === 'keywords') {
                    // Remove quotes and convert string representation of array to array
                    $value = str_replace(['[', ']', "'", '"'], '', $value);
                    $value = array_map('trim', explode(',', $value));
                } elseif ($columnName === 'popular') {
                    $value = ($value === 'True' || $value === 'true' || $value === '1');
                } elseif ($columnName === 'price') {
                    $value = (float)$value;
                } elseif ($columnName === 'id') {
                    $value = (int)$value;
                }

                $product[$columnName] = $value;
            }

            $products[] = $product;
        }

        fclose($handle);
        return $products;
    }

    /**
     * Cache products to JSON file
     */
    private function cacheProducts(array $products): void {
        file_put_contents($this->cacheFile, json_encode($products, JSON_PRETTY_PRINT));
    }

    /**
     * Check if cache is still valid
     */
    private function isCacheValid(): bool {
        if (!file_exists($this->cacheFile)) {
            return false;
        }

        $fileAge = time() - filemtime($this->cacheFile);
        return $fileAge < $this->cacheDuration;
    }

    /**
     * Clear cache (call this when CSV is updated)
     */
    public function clearCache(): void {
        if (file_exists($this->cacheFile)) {
            unlink($this->cacheFile);
        }
    }

    /**
     * Force reload from CSV
     */
    public function reloadFromCSV(): array {
        $this->clearCache();
        return $this->getAllProducts();
    }
}

// ============================================
// END OF CSV READER CLASS
// ============================================
