<?php
/**
 * CDVectors Database Connection Class
 * ====================================
 * Handles all database operations with prepared statements
 */

require_once __DIR__ . '/config.php';

class Database {
    private ?mysqli $connection = null;
    private static ?Database $instance = null;

    /**
     * Private constructor to prevent direct instantiation
     */
    private function __construct() {
        $this->connect();
    }

    /**
     * Singleton pattern - get database instance
     */
    public static function getInstance(): Database {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Establish database connection
     */
    private function connect(): void {
        // Create connection
        $this->connection = new mysqli(
            DB_HOST,
            DB_USER,
            DB_PASS,
            DB_NAME,
            DB_PORT
        );

        // Check connection
        if ($this->connection->connect_error) {
            error_log('Database connection error: ' . $this->connection->connect_error);
            if (APP_ENV === 'development') {
                die('Connection failed: ' . $this->connection->connect_error);
            } else {
                die('Database connection error. Please try again later.');
            }
        }

        // Set charset to utf8mb4
        if (!$this->connection->set_charset(DB_CHARSET)) {
            error_log('Error loading charset: ' . $this->connection->error);
        }
    }

    /**
     * Get raw connection (use with caution)
     */
    public function getConnection(): mysqli {
        if ($this->connection === null || !$this->connection->ping()) {
            $this->connect();
        }
        return $this->connection;
    }

    /**
     * Execute prepared statement with parameters
     * @param string $query SQL query with ? placeholders
     * @param array $params Parameters to bind
     * @param string $types Data types (i=int, d=double, s=string, b=blob)
     * @return mysqli_result|bool
     */
    public function query(string $query, array $params = [], string $types = ''): mysqli_result|bool {
        try {
            $stmt = $this->getConnection()->prepare($query);

            if (!$stmt) {
                throw new Exception('Prepare failed: ' . $this->getConnection()->error);
            }

            // Bind parameters if provided
            if (!empty($params)) {
                if (empty($types)) {
                    // Auto-detect types if not provided
                    $types = $this->detectTypes($params);
                }
                $stmt->bind_param($types, ...$params);
            }

            // Execute statement
            if (!$stmt->execute()) {
                throw new Exception('Execute failed: ' . $stmt->error);
            }

            return $stmt->get_result();
        } catch (Exception $e) {
            error_log('Database query error: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Get single row
     */
    public function getRow(string $query, array $params = []): ?array {
        $result = $this->query($query, $params);
        return $result ? $result->fetch_assoc() : null;
    }

    /**
     * Get all rows
     */
    public function getRows(string $query, array $params = []): array {
        $result = $this->query($query, $params);
        if (!$result) return [];

        $rows = [];
        while ($row = $result->fetch_assoc()) {
            $rows[] = $row;
        }
        $result->free();
        return $rows;
    }

    /**
     * Insert data (returns last insert ID)
     */
    public function insert(string $table, array $data): int|false {
        $columns = array_keys($data);
        $values = array_values($data);
        $placeholders = implode(',', array_fill(0, count($columns), '?'));
        $columnsList = implode(',', $columns);

        $query = "INSERT INTO `$table` ($columnsList) VALUES ($placeholders)";
        
        $types = $this->detectTypes($values);
        
        try {
            $this->query($query, $values, $types);
            return $this->getConnection()->insert_id;
        } catch (Exception $e) {
            error_log('Insert error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Update data (returns number of affected rows)
     */
    public function update(string $table, array $data, array $where): int|false {
        if (empty($data) || empty($where)) {
            return false;
        }

        // Build SET clause
        $setClause = implode(',', array_map(fn($col) => "`$col`=?", array_keys($data)));
        
        // Build WHERE clause
        $whereClause = implode(' AND ', array_map(fn($col) => "`$col`=?", array_keys($where)));

        // Merge values
        $values = array_merge(array_values($data), array_values($where));
        $types = $this->detectTypes($values);

        $query = "UPDATE `$table` SET $setClause WHERE $whereClause";

        try {
            $stmt = $this->getConnection()->prepare($query);
            $stmt->bind_param($types, ...$values);
            $stmt->execute();
            return $this->getConnection()->affected_rows;
        } catch (Exception $e) {
            error_log('Update error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Delete data
     */
    public function delete(string $table, array $where): int|false {
        if (empty($where)) {
            return false;
        }

        $whereClause = implode(' AND ', array_map(fn($col) => "`$col`=?", array_keys($where)));
        $values = array_values($where);
        $types = $this->detectTypes($values);

        $query = "DELETE FROM `$table` WHERE $whereClause";

        try {
            $stmt = $this->getConnection()->prepare($query);
            $stmt->bind_param($types, ...$values);
            $stmt->execute();
            return $this->getConnection()->affected_rows;
        } catch (Exception $e) {
            error_log('Delete error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Count rows
     */
    public function count(string $table, array $where = []): int {
        if (empty($where)) {
            $query = "SELECT COUNT(*) as count FROM `$table`";
            $result = $this->query($query);
        } else {
            $whereClause = implode(' AND ', array_map(fn($col) => "`$col`=?", array_keys($where)));
            $values = array_values($where);
            $query = "SELECT COUNT(*) as count FROM `$table` WHERE $whereClause";
            $result = $this->query($query, $values);
        }

        $row = $result->fetch_assoc();
        return $row['count'] ?? 0;
    }

    /**
     * Auto-detect parameter types
     */
    private function detectTypes(array $params): string {
        $types = '';
        foreach ($params as $param) {
            if (is_int($param)) {
                $types .= 'i';
            } elseif (is_float($param)) {
                $types .= 'd';
            } elseif (is_string($param)) {
                $types .= 's';
            } else {
                $types .= 's'; // Default to string
            }
        }
        return $types;
    }

    /**
     * Begin transaction
     */
    public function beginTransaction(): bool {
        return $this->getConnection()->begin_transaction();
    }

    /**
     * Commit transaction
     */
    public function commit(): bool {
        return $this->getConnection()->commit();
    }

    /**
     * Rollback transaction
     */
    public function rollback(): bool {
        return $this->getConnection()->rollback();
    }

    /**
     * Get last error
     */
    public function getLastError(): string {
        return $this->getConnection()->error ?? 'Unknown error';
    }

    /**
     * Close connection
     */
    public function close(): void {
        if ($this->connection) {
            $this->connection->close();
            $this->connection = null;
        }
    }

    /**
     * Get last inserted ID
     */
    public function getInsertId(): int {
        return (int)$this->getConnection()->insert_id;
    }

    /**
     * Escape string (for safety)
     */
    public function escape(string $string): string {
        return $this->getConnection()->real_escape_string($string);
    }
}

// ============================================
// Create logs directory if not exists
// ============================================
if (!is_dir(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

// ============================================
// END OF DATABASE CLASS
// ============================================
