<?php
/**
 * User Login Page
 */

// Set page title
$pageTitle = 'Giriş Yap';
$pageDescription = 'CDVectors hesabınıza giriş yapın.';

require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/auth.php';

// If already logged in, redirect based on redirect_to or home
if (isset($_SESSION['user_id'])) {
    $redirectTo = $_SESSION['redirect_to'] ?? APP_URL;
    unset($_SESSION['redirect_to']);
    header('Location: ' . $redirectTo);
    exit();
}

$auth = new Auth();
$errors = [];

// Process login form
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !Auth::verifyCsrfToken($_POST['csrf_token'])) {
        $errors[] = 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.';
    } else {
        $username = $_POST['username'] ?? '';
        $password = $_POST['password'] ?? '';

        $result = $auth->login($username, $password);
        if ($result['success']) {
            $_SESSION['flash_message'] = $result['message'];
            $_SESSION['flash_type'] = 'success';
            
            // Redirect to previous page or home
            $redirectTo = $_SESSION['redirect_to'] ?? APP_URL . '/dashboard.php';
            unset($_SESSION['redirect_to']);
            
            header('Location: ' . $redirectTo);
            exit();
        } else {
            $errors[] = $result['message'];
        }
    }
}

// Generate CSRF token for form
$csrfToken = Auth::generateCsrfToken();
?>

<div class="auth-container">
    <div class="auth-box login-box">
        <h1>Giriş Yap</h1>
        <p class="auth-subtitle">Hesabınıza giriş yaparak devam edin</p>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-error" role="alert">
                <?php foreach ($errors as $error): ?>
                    <p><?php echo htmlspecialchars($error); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <form method="POST" class="auth-form" action="<?php echo APP_URL; ?>/login.php">
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo $csrfToken; ?>">

            <!-- Username or Email -->
            <div class="form-group">
                <label for="username">Kullanıcı Adı veya E-posta *</label>
                <input 
                    type="text" 
                    id="username" 
                    name="username" 
                    placeholder="Kullanıcı adı veya e-posta"
                    value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>"
                    required
                    autocomplete="username"
                    autofocus
                >
            </div>

            <!-- Password -->
            <div class="form-group">
                <label for="password">Şifre *</label>
                <input 
                    type="password" 
                    id="password" 
                    name="password" 
                    placeholder="••••••••"
                    required
                    autocomplete="current-password"
                >
            </div>

            <!-- Remember Me -->
            <div class="form-group checkbox">
                <input 
                    type="checkbox" 
                    id="remember" 
                    name="remember"
                >
                <label for="remember">Beni hatırla</label>
            </div>

            <!-- Submit Button -->
            <button type="submit" class="btn btn-primary btn-block">
                Giriş Yap
            </button>
        </form>

        <!-- Footer Links -->
        <div class="auth-links">
            <a href="<?php echo APP_URL; ?>/forgot-password.php" class="link">Şifremi unuttum</a>
            <span class="divider">•</span>
            <a href="<?php echo APP_URL; ?>/register.php" class="link">Yeni hesap oluştur</a>
        </div>

        <!-- Footer Text -->
        <p class="auth-footer">
            Hesabınız yok mu? 
            <a href="<?php echo APP_URL; ?>/register.php">Şimdi kayıt olun</a>
        </p>
    </div>
</div>

<style>
.auth-container {
    display: flex;
    justify-content: center;
    align-items: center;
    min-height: calc(100vh - 200px);
    padding: 20px;
}

.auth-box {
    background: white;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    padding: 40px;
    max-width: 400px;
    width: 100%;
}

.auth-box h1 {
    margin: 0 0 10px 0;
    color: #333;
    font-size: 24px;
}

.auth-subtitle {
    color: #666;
    margin: 0 0 20px 0;
    font-size: 14px;
}

.auth-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    margin-bottom: 5px;
    font-weight: 500;
    color: #333;
}

.form-group input {
    padding: 10px 12px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
    transition: border-color 0.3s;
}

.form-group input:focus {
    outline: none;
    border-color: #007bff;
    box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
}

.form-group.checkbox {
    flex-direction: row;
    align-items: center;
    gap: 8px;
}

.form-group.checkbox input {
    width: auto;
}

.form-group.checkbox label {
    margin: 0;
    font-weight: normal;
    font-size: 14px;
}

.btn {
    padding: 10px 15px;
    border: none;
    border-radius: 4px;
    font-weight: 500;
    cursor: pointer;
    transition: background-color 0.3s;
}

.btn-primary {
    background-color: #007bff;
    color: white;
}

.btn-primary:hover {
    background-color: #0056b3;
}

.btn-block {
    width: 100%;
}

.auth-links {
    text-align: center;
    margin: 20px 0;
    font-size: 14px;
}

.auth-links .link {
    color: #007bff;
    text-decoration: none;
}

.auth-links .link:hover {
    text-decoration: underline;
}

.auth-links .divider {
    color: #ddd;
    margin: 0 8px;
}

.auth-footer {
    text-align: center;
    margin-top: 20px;
    color: #666;
    font-size: 14px;
}

.auth-footer a {
    color: #007bff;
    text-decoration: none;
}

.auth-footer a:hover {
    text-decoration: underline;
}

.alert {
    padding: 15px;
    border-radius: 4px;
    margin-bottom: 20px;
}

.alert-error {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.alert p {
    margin: 0;
}

.alert p + p {
    margin-top: 8px;
}
</style>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
