<?php
/**
 * My Downloads Page
 * Shows files available for download based on orders
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/database.php';
require_once __DIR__ . '/includes/csv-reader.php';

$db = Database::getInstance();
$csvReader = new CSVReader();
$pageTitle = 'Dosyalarım';

require_once __DIR__ . '/includes/header.php';

// Get order ID from URL parameter
$orderId = isset($_GET['order_id']) ? (int)$_GET['order_id'] : 0;

$order = null;
$orderItems = [];

if ($orderId > 0) {
    // Fetch order details
    $orderQuery = "SELECT * FROM orders WHERE id = ?";
    $orderResult = $db->query($orderQuery, [$orderId], 'i');
    
    if ($orderResult && $orderResult->num_rows > 0) {
        $order = $orderResult->fetch_assoc();
        
        // Fetch order items with product details
        $itemsQuery = "
            SELECT oi.*, p.title, p.price, p.image 
            FROM order_items oi
            LEFT JOIN products p ON oi.product_id = p.id
            WHERE oi.order_id = ?
        ";
        $itemsResult = $db->query($itemsQuery, [$orderId], 'i');
        
        if ($itemsResult) {
            while ($row = $itemsResult->fetch_assoc()) {
                $row['product_data'] = $csvReader->getProductById($row['product_id']);
                $orderItems[] = $row;
            }
        }
    }
}

?>

<div style="max-width: 1000px; margin: 0 auto; padding: 2rem 0;">
    
    <h1>📥 Dosyalarım</h1>

    <?php if ($order): ?>
        <!-- Order Info -->
        <div style="
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: 0 2px 12px rgba(0,0,0,0.08);
            margin-bottom: 2rem;
            border-left: 4px solid #ff6b9d;
        ">
            <h2 style="margin-top: 0;">Sipariş #<?php echo htmlspecialchars($order['order_code']); ?></h2>
            <p style="color: #999; margin-bottom: 1rem;">
                Sipariş Tarihi: <?php echo date('d.m.Y H:i', strtotime($order['created_at'])); ?>
            </p>
            
            <?php
            // Check if download link is still valid (1 hour after order)
            $orderTime = strtotime($order['created_at']);
            $currentTime = time();
            $timeDiff = $currentTime - $orderTime;
            $oneHour = 3600;
            $isValid = $timeDiff <= $oneHour;
            $minutesLeft = floor(($oneHour - $timeDiff) / 60);
            ?>

            <p style="
                background: <?php echo $isValid ? '#d4edda' : '#f8d7da'; ?>;
                color: <?php echo $isValid ? '#155724' : '#721c24'; ?>;
                padding: 1rem;
                border-radius: 8px;
                border-left: 4px solid <?php echo $isValid ? '#28a745' : '#f5222d'; ?>;
            ">
                <?php if ($isValid): ?>
                    ✅ <strong>İndirme Bağlantıları Aktif</strong><br>
                    Dosyaları indirebileceğiniz kalan süre: <strong><?php echo $minutesLeft; ?> dakika</strong>
                <?php else: ?>
                    ⚠️ <strong>İndirme Süresi Doldu</strong><br>
                    Dosya indirme bağlantılarınız 1 saat sonra geçersiz hale gelir. 
                    <a href="<?php echo APP_URL; ?>/support.php">Destek ekibimize başvurun</a> yeni bağlantı almak için.
                <?php endif; ?>
            </p>
        </div>

        <!-- Download Items -->
        <?php if (!empty($orderItems)): ?>
            <div style="margin-bottom: 2rem;">
                <h3>📦 Satın Alınan Ürünler</h3>
                
                <?php foreach ($orderItems as $item): ?>
                    <?php
                    $product = $item['product_data'];
                    if (!$product) continue;
                    ?>
                    <div style="
                        background: white;
                        padding: 2rem;
                        border-radius: 12px;
                        box-shadow: 0 2px 12px rgba(0,0,0,0.08);
                        margin-bottom: 1.5rem;
                        display: grid;
                        grid-template-columns: 120px 1fr;
                        gap: 2rem;
                        align-items: center;
                    ">
                        <!-- Product Image -->
                        <div>
                            <img 
                                src="<?php echo htmlspecialchars($product['image']); ?>" 
                                alt="<?php echo htmlspecialchars($product['title']); ?>"
                                style="
                                    width: 100%;
                                    height: 120px;
                                    object-fit: cover;
                                    border-radius: 8px;
                                "
                            >
                        </div>

                        <!-- Product Details -->
                        <div>
                            <h4 style="margin-top: 0; margin-bottom: 0.5rem;">
                                <?php echo htmlspecialchars($product['title']); ?>
                            </h4>
                            <p style="color: #999; margin: 0.25rem 0;">
                                Kategori: <strong><?php echo htmlspecialchars(ucfirst($product['category'])); ?></strong>
                            </p>
                            <p style="color: #999; margin: 0.25rem 0;">
                                Fiyat: <strong>$<?php echo number_format($product['price'], 2); ?></strong>
                            </p>
                            <p style="color: #999; margin: 0.25rem 0;">
                                Miktar: <strong><?php echo htmlspecialchars($item['quantity']); ?></strong> x
                            </p>

                            <!-- Download Button -->
                            <div style="margin-top: 1.5rem;">
                                <?php if ($isValid): ?>
                                    <a 
                                        href="<?php echo APP_URL; ?>/api/downloads/get.php?product_id=<?php echo $product['id']; ?>&order_id=<?php echo $orderId; ?>"
                                        style="
                                            display: inline-block;
                                            background: linear-gradient(135deg, #ff6b9d 0%, #e63946 100%);
                                            color: white;
                                            padding: 0.75rem 1.5rem;
                                            border-radius: 8px;
                                            text-decoration: none;
                                            font-weight: 600;
                                            transition: all 0.3s ease;
                                        "
                                        onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 8px 24px rgba(230, 57, 70, 0.3)';"
                                        onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none';"
                                    >
                                        💾 İndir
                                    </a>
                                <?php else: ?>
                                    <button 
                                        disabled
                                        style="
                                            display: inline-block;
                                            background: #ccc;
                                            color: #999;
                                            padding: 0.75rem 1.5rem;
                                            border-radius: 8px;
                                            border: none;
                                            font-weight: 600;
                                            cursor: not-allowed;
                                        "
                                    >
                                        ❌ İndirme Süresi Doldu
                                    </button>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else: ?>
            <div style="
                background: #fff3cd;
                padding: 2rem;
                border-radius: 12px;
                text-align: center;
                margin-bottom: 2rem;
            ">
                <p style="color: #666; margin: 0;">
                    Bu siparişte hiçbir ürün bulunamadı.
                </p>
            </div>
        <?php endif; ?>

        <!-- Additional Info -->
        <div style="
            background: #f0f8ff;
            border: 1px solid #87ceeb;
            padding: 2rem;
            border-radius: 12px;
            margin-bottom: 2rem;
        ">
            <h3 style="margin-top: 0;">ℹ️ İndirme Bilgileri</h3>
            <ul style="color: #333;">
                <li style="margin-bottom: 1rem;">
                    <strong>İndirme Süresi:</strong> Sipariş sonrası 1 saat boyunca indirebilirsiniz.
                </li>
                <li style="margin-bottom: 1rem;">
                    <strong>Format:</strong> Tüm dosyalar ZIP formatında verildiğinde farklı formatları içerebilir (AI, EPS, PDF, PNG, SVG, vb.).
                </li>
                <li style="margin-bottom: 1rem;">
                    <strong>Tekrar İndirme:</strong> Dosya indirme süreniz dolmuşsa 
                    <a href="<?php echo APP_URL; ?>/support.php">destek ekibimize</a> başvurun.
                </li>
                <li>
                    <strong>Sorun Yaşıyorsanız:</strong> Dosya indirmede sorun yaşarsanız, 
                    lütfen <a href="<?php echo APP_URL; ?>/support.php">iletişim formunu</a> doldurun.
                </li>
            </ul>
        </div>

        <!-- Back to Account -->
        <div style="text-align: center;">
            <a 
                href="<?php echo APP_URL; ?>/account.php"
                style="
                    display: inline-block;
                    background: #f5f5f5;
                    color: #666;
                    padding: 1rem 2rem;
                    border-radius: 8px;
                    text-decoration: none;
                    font-weight: 600;
                    transition: all 0.3s ease;
                "
                onmouseover="this.style.backgroundColor='#efefef';"
                onmouseout="this.style.backgroundColor='#f5f5f5';"
            >
                ← Hesabıma Dön
            </a>
        </div>

    <?php else: ?>
        <!-- No Order Found -->
        <div style="
            background: white;
            padding: 3rem 2rem;
            border-radius: 12px;
            text-align: center;
            box-shadow: 0 2px 12px rgba(0,0,0,0.08);
        ">
            <div style="font-size: 2rem; margin-bottom: 1rem;">❓</div>
            <h2>Siparış Bulunamadı</h2>
            <p style="color: #999; margin-bottom: 2rem;">
                Maalesef, talep ettiğiniz siparış bulunamadı. 
                Lütfen siparış numarasını kontrol edin veya destek ekibimize başvurun.
            </p>
            <a 
                href="<?php echo APP_URL; ?>/support.php"
                style="
                    display: inline-block;
                    background: linear-gradient(135deg, #ff6b9d 0%, #e63946 100%);
                    color: white;
                    padding: 1rem 2rem;
                    border-radius: 8px;
                    text-decoration: none;
                    font-weight: 600;
                    transition: all 0.3s ease;
                "
                onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 8px 24px rgba(230, 57, 70, 0.3)';"
                onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none';"
            >
                Destek Ekibine Ulaş
            </a>
        </div>
    <?php endif; ?>
</div>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
