<?php
/**
 * Product Detail Page
 * Shows detailed information about a single product
 */

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/csv-reader.php';

$csvReader = new CSVReader();

// Get product ID from URL
$productId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($productId <= 0) {
    header('Location: ' . APP_URL . '/products.php');
    exit();
}

// Get product data
$product = $csvReader->getProductById($productId);

if (!$product) {
    header('HTTP/1.0 404 Not Found');
    $pageTitle = 'Ürün Bulunamadı';
    require_once __DIR__ . '/includes/header.php';
    ?>
    <div style="text-align: center; padding: 3rem;">
        <h2>Ürün Bulunamadı 😢</h2>
        <p>Aradığınız ürün mevcut değil.</p>
        <a href="<?php echo APP_URL; ?>/products.php" class="btn btn-primary" style="margin-top: 1rem;">
            Tüm Ürünlere Dön
        </a>
    </div>
    <?php
    require_once __DIR__ . '/includes/footer.php';
    exit();
}

$pageTitle = htmlspecialchars($product['title']);
$pageDescription = htmlspecialchars(substr($product['description'], 0, 160));

require_once __DIR__ . '/includes/header.php';

// Get related products (same category)
$relatedProducts = array_slice($csvReader->getProductsByCategory($product['category']), 0, 4);
?>

<div style="max-width: 1200px; margin: 0 auto; padding: 2rem 0;">
    <!-- Breadcrumb -->
    <div style="margin-bottom: 2rem; color: #666; font-size: 0.9rem;">
        <a href="<?php echo APP_URL; ?>">Anasayfa</a> / 
        <a href="<?php echo APP_URL; ?>/products.php?category=<?php echo htmlspecialchars($product['category']); ?>">
            <?php echo htmlspecialchars(ucfirst($product['category'])); ?>
        </a> / 
        <span><?php echo htmlspecialchars($product['title']); ?></span>
    </div>

    <!-- Product Details Grid -->
    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 3rem; margin-bottom: 3rem;">
        <!-- Left: Product Image -->
        <div>
            <div style="
                background: white;
                border-radius: 12px;
                overflow: hidden;
                box-shadow: 0 2px 12px rgba(230, 57, 70, 0.1);
                margin-bottom: 1rem;
            ">
                <img 
                    src="<?php echo htmlspecialchars($product['image']); ?>" 
                    alt="<?php echo htmlspecialchars($product['title']); ?>"
                    style="width: 100%; height: auto; display: block;"
                >
            </div>
            <p style="text-align: center; color: #999; font-size: 0.9rem;">
                💝 Premium Vektör Grafikleri
            </p>
        </div>

        <!-- Right: Product Info -->
        <div>
            <!-- Title -->
            <h1 style="margin-bottom: 1rem; color: #2d2d2d;">
                <?php echo htmlspecialchars($product['title']); ?>
            </h1>

            <!-- Category & Rating -->
            <div style="margin-bottom: 1.5rem; display: flex; gap: 1rem; align-items: center;">
                <span style="
                    display: inline-block;
                    background: #ff6b9d;
                    color: white;
                    padding: 0.5rem 1rem;
                    border-radius: 20px;
                    font-size: 0.9rem;
                    font-weight: 600;
                ">
                    <?php echo htmlspecialchars(ucfirst($product['category'])); ?>
                </span>
                <span style="color: #ffc107;">⭐⭐⭐⭐⭐ (<?php echo htmlspecialchars($product['customers']) ? count(explode(',', $product['customers'])) : 0; ?> değerlendirme)</span>
            </div>

            <!-- Price -->
            <div style="margin-bottom: 2rem;">
                <p style="color: #999; font-size: 0.9rem;">Fiyat</p>
                <h2 style="
                    background: linear-gradient(135deg, #ff6b9d 0%, #e63946 100%);
                    -webkit-background-clip: text;
                    -webkit-text-fill-color: transparent;
                    background-clip: text;
                    margin: 0.5rem 0;
                ">
                    $<?php echo number_format($product['price'], 2); ?>
                </h2>
            </div>

            <!-- Description -->
            <div style="margin-bottom: 2rem; line-height: 1.8;">
                <h3 style="margin-bottom: 1rem;">Ürün Açıklaması</h3>
                <p style="color: #666;">
                    <?php echo htmlspecialchars($product['description']); ?>
                </p>
            </div>

            <!-- File Info -->
            <?php if (isset($product['file_type']) && $product['file_type']): ?>
                <div style="
                    background: #faf8f6;
                    padding: 1.5rem;
                    border-radius: 8px;
                    margin-bottom: 2rem;
                    border-left: 4px solid #ff6b9d;
                ">
                    <h4 style="margin-bottom: 0.5rem;">📦 Dosya Bilgileri</h4>
                    <p><strong>Tür:</strong> <?php echo htmlspecialchars(strtoupper($product['file_type'])); ?></p>
                    <p><strong>Boyut:</strong> <?php echo htmlspecialchars($product['file_size'] ?? 'Belirtilmemiş'); ?></p>
                </div>
            <?php endif; ?>

            <!-- Actions -->
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 1rem; margin-bottom: 2rem;">
                <button 
                    onclick="addToCart(<?php echo $product['id']; ?>, '<?php echo htmlspecialchars($product['title']); ?>')"
                    style="
                        background: linear-gradient(135deg, #ff6b9d 0%, #e63946 100%);
                        color: white;
                        border: none;
                        padding: 1rem;
                        border-radius: 8px;
                        font-size: 1.1rem;
                        font-weight: 600;
                        cursor: pointer;
                        transition: all 0.3s ease;
                    "
                    onmouseover="this.style.transform='translateY(-3px)'; this.style.boxShadow='0 8px 24px rgba(230, 57, 70, 0.3)';"
                    onmouseout="this.style.transform='translateY(0)'; this.style.boxShadow='none';"
                >
                    🛒 Sepete Ekle
                </button>
                <button 
                    onclick="addToWishlist(<?php echo $product['id']; ?>)"
                    style="
                        background: white;
                        color: #e63946;
                        border: 2px solid #e63946;
                        padding: 1rem;
                        border-radius: 8px;
                        font-size: 1.1rem;
                        font-weight: 600;
                        cursor: pointer;
                        transition: all 0.3s ease;
                    "
                    onmouseover="this.style.backgroundColor='#e63946'; this.style.color='white';"
                    onmouseout="this.style.backgroundColor='white'; this.style.color='#e63946';"
                >
                    ❤️ Favorilere Ekle
                </button>
            </div>

            <!-- Tags -->
            <?php if (isset($product['tags']) && is_array($product['tags']) && count($product['tags']) > 0): ?>
                <div style="margin-bottom: 2rem;">
                    <h4 style="margin-bottom: 1rem;">Etiketler</h4>
                    <div style="display: flex; flex-wrap: wrap; gap: 0.75rem;">
                        <?php foreach (array_slice($product['tags'], 0, 8) as $tag): ?>
                            <a 
                                href="<?php echo APP_URL; ?>/products.php?search=<?php echo htmlspecialchars($tag); ?>"
                                style="
                                    background: #f4a0a0;
                                    color: white;
                                    padding: 0.5rem 1rem;
                                    border-radius: 20px;
                                    font-size: 0.85rem;
                                    text-decoration: none;
                                    transition: all 0.3s ease;
                                "
                                onmouseover="this.style.background='#ff6b9d';"
                                onmouseout="this.style.background='#f4a0a0';"
                            >
                                #<?php echo htmlspecialchars($tag); ?>
                            </a>
                        <?php endforeach; ?>
                    </div>
                </div>
            <?php endif; ?>

            <!-- Share -->
            <div style="border-top: 1px solid #eee; padding-top: 1.5rem;">
                <p style="font-size: 0.9rem; color: #999;">Bu ürünü sosyal medyada paylaş:</p>
                <div style="display: flex; gap: 1rem; margin-top: 0.5rem;">
                    <a href="#" style="font-size: 1.5rem; text-decoration: none;">📘</a>
                    <a href="#" style="font-size: 1.5rem; text-decoration: none;">🐦</a>
                    <a href="#" style="font-size: 1.5rem; text-decoration: none;">📌</a>
                </div>
            </div>
        </div>
    </div>

    <!-- Related Products -->
    <?php if (!empty($relatedProducts)): ?>
        <div style="margin-top: 4rem; padding-top: 2rem; border-top: 2px solid #eee;">
            <h2 style="margin-bottom: 2rem; text-align: center;">
                İlgili Ürünler 💕
            </h2>
            <div class="products-grid">
                <?php foreach ($relatedProducts as $related): ?>
                    <?php if ($related['id'] !== $product['id']): ?>
                        <div class="product-card fade-in">
                            <a href="<?php echo APP_URL; ?>/product-detail.php?id=<?php echo $related['id']; ?>">
                                <div class="product-image">
                                    <img 
                                        src="<?php echo htmlspecialchars($related['image']); ?>" 
                                        alt="<?php echo htmlspecialchars($related['title']); ?>"
                                        loading="lazy"
                                    >
                                </div>
                            </a>
                            <div class="product-info">
                                <span class="product-category">
                                    <?php echo htmlspecialchars(ucfirst($related['category'])); ?>
                                </span>
                                <h3 class="product-title">
                                    <a href="<?php echo APP_URL; ?>/product-detail.php?id=<?php echo $related['id']; ?>">
                                        <?php echo htmlspecialchars(substr($related['title'], 0, 50)); ?>
                                    </a>
                                </h3>
                                <div class="product-footer">
                                    <span class="product-price">$<?php echo number_format($related['price'], 2); ?></span>
                                    <button 
                                        class="btn-add-cart"
                                        onclick="addToCart(<?php echo $related['id']; ?>, '<?php echo htmlspecialchars($related['title']); ?>')"
                                    >
                                        🛒 Ekle
                                    </button>
                                </div>
                            </div>
                        </div>
                    <?php endif; ?>
                <?php endforeach; ?>
            </div>
        </div>
    <?php endif; ?>
</div>

<!-- JavaScript -->
<script>
function addToCart(productId, productTitle) {
    let cart = JSON.parse(localStorage.getItem('cart')) || [];
    const existingItem = cart.find(item => item.id === productId);

    if (existingItem) {
        existingItem.quantity += 1;
    } else {
        cart.push({
            id: productId,
            title: productTitle,
            quantity: 1
        });
    }

    localStorage.setItem('cart', JSON.stringify(cart));
    updateCartCount();
    alert(productTitle + ' sepete eklendi! 🛒');
}

function addToWishlist(productId) {
    let wishlist = JSON.parse(localStorage.getItem('wishlist')) || [];
    
    if (!wishlist.find(item => item.id === productId)) {
        wishlist.push({
            id: productId,
            addedAt: new Date().toISOString()
        });
        localStorage.setItem('wishlist', JSON.stringify(wishlist));
        alert('Ürün favorilere eklendi! ❤️');
    } else {
        alert('Bu ürün zaten favorilerin içinde!');
    }
}

function updateCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart')) || [];
    const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
    document.getElementById('cart-count').textContent = totalItems;
}

window.addEventListener('DOMContentLoaded', updateCartCount);
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
