<?php
/**
 * Products Listing Page
 * Lists all products from CSV with filtering and search
 */

$pageTitle = 'Tüm Ürünler';
$pageDescription = 'Premium vektör grafikleri ve tasarım şablonlarını keşfedin';

require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/csv-reader.php';

$csvReader = new CSVReader();

// Get query parameters
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$category = isset($_GET['category']) ? htmlspecialchars($_GET['category']) : null;
$search = isset($_GET['search']) ? htmlspecialchars($_GET['search']) : null;
$perPage = 12;

// Determine what products to show
if ($search) {
    // Search results
    $allProducts = $csvReader->searchProducts($search);
    $totalProducts = count($allProducts);
    $totalPages = ceil($totalProducts / $perPage);
    $page = max(1, min($page, $totalPages));
    $startIndex = ($page - 1) * $perPage;
    $products = array_slice($allProducts, $startIndex, $perPage);
    $pageTitle = "Arama Sonuçları: " . htmlspecialchars($search);
} elseif ($category) {
    // Category filter
    $data = $csvReader->getProductsPaginated($page, $perPage, $category);
    $products = $data['products'];
    $totalPages = $data['totalPages'];
    $totalProducts = $data['totalProducts'];
    $pageTitle = ucfirst($category) . " Kategorisi";
} else {
    // All products
    $data = $csvReader->getProductsPaginated($page, $perPage);
    $products = $data['products'];
    $totalPages = $data['totalPages'];
    $totalProducts = $data['totalProducts'];
}

// Get categories for filter
$categories = $csvReader->getCategories();
?>

<div class="products-section">
    <!-- Section Header -->
    <div class="section-title">
        <h2><?php echo htmlspecialchars($pageTitle); ?></h2>
        <p><?php echo $totalProducts; ?> ürün bulundu</p>
    </div>

    <!-- Filters & Search -->
    <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 1rem; margin-bottom: 2rem;">
        <!-- Search -->
        <form method="GET" style="display: flex; gap: 0.5rem;">
            <input type="text" name="search" placeholder="Ürün ara..." value="<?php echo $search ?? ''; ?>" style="flex: 1;">
            <button type="submit" class="btn btn-primary">Ara</button>
        </form>

        <!-- Category Filter -->
        <form method="GET">
            <select name="category" onchange="this.form.submit()" style="width: 100%;">
                <option value="">Tüm Kategoriler</option>
                <?php foreach ($categories as $cat): ?>
                    <option value="<?php echo htmlspecialchars($cat); ?>" <?php echo ($category === $cat) ? 'selected' : ''; ?>>
                        <?php echo htmlspecialchars(ucfirst($cat)); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </form>

        <!-- Clear Filters -->
        <div>
            <?php if ($search || $category): ?>
                <a href="<?php echo APP_URL; ?>/products.php" class="btn btn-secondary" style="width: 100%; text-align: center;">
                    Filtreleri Temizle ✕
                </a>
            <?php endif; ?>
        </div>
    </div>

    <!-- Products Grid -->
    <?php if (!empty($products)): ?>
        <div class="products-grid">
            <?php foreach ($products as $product): ?>
                <div class="product-card fade-in">
                    <!-- Product Image -->
                    <a href="<?php echo APP_URL; ?>/product-detail.php?id=<?php echo $product['id']; ?>">
                        <div class="product-image">
                            <img 
                                src="<?php echo htmlspecialchars($product['image']); ?>" 
                                alt="<?php echo htmlspecialchars($product['title']); ?>"
                                loading="lazy"
                            >
                        </div>
                    </a>

                    <!-- Product Info -->
                    <div class="product-info">
                        <!-- Category Badge -->
                        <span class="product-category">
                            <?php echo htmlspecialchars(ucfirst($product['category'])); ?>
                        </span>

                        <!-- Title -->
                        <h3 class="product-title">
                            <a href="<?php echo APP_URL; ?>/product-detail.php?id=<?php echo $product['id']; ?>">
                                <?php echo htmlspecialchars(substr($product['title'], 0, 50)); ?>
                            </a>
                        </h3>

                        <!-- Description (truncated) -->
                        <p class="product-description">
                            <?php 
                            $desc = htmlspecialchars(substr($product['description'], 0, 100));
                            echo $desc . "...";
                            ?>
                        </p>

                        <!-- Footer: Price & Button -->
                        <div class="product-footer">
                            <span class="product-price">$<?php echo number_format($product['price'], 2); ?></span>
                            <button class="btn-add-cart" onclick="addToCart(<?php echo $product['id']; ?>, '<?php echo htmlspecialchars($product['title']); ?>')">
                                🛒 Ekle
                            </button>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Pagination -->
        <?php if ($totalPages > 1): ?>
            <div class="pagination">
                <?php
                // Previous button
                if ($page > 1) {
                    $prevUrl = $_SERVER['REQUEST_URI'];
                    $prevUrl = preg_replace('/([?&])page=\d+/', '', $prevUrl);
                    $prevUrl .= (strpos($prevUrl, '?') ? '&' : '?') . 'page=' . ($page - 1);
                    echo '<a href="' . htmlspecialchars($prevUrl) . '">← Önceki</a>';
                }

                // Page numbers
                $start = max(1, $page - 2);
                $end = min($totalPages, $page + 2);

                if ($start > 1) {
                    echo '<a href="?page=1">1</a>';
                    if ($start > 2) echo '<span>...</span>';
                }

                for ($i = $start; $i <= $end; $i++) {
                    if ($i == $page) {
                        echo '<span class="current">' . $i . '</span>';
                    } else {
                        $pageUrl = $_SERVER['REQUEST_URI'];
                        $pageUrl = preg_replace('/([?&])page=\d+/', '', $pageUrl);
                        $pageUrl .= (strpos($pageUrl, '?') ? '&' : '?') . 'page=' . $i;
                        echo '<a href="' . htmlspecialchars($pageUrl) . '">' . $i . '</a>';
                    }
                }

                if ($end < $totalPages) {
                    if ($end < $totalPages - 1) echo '<span>...</span>';
                    echo '<a href="?page=' . $totalPages . '">' . $totalPages . '</a>';
                }

                // Next button
                if ($page < $totalPages) {
                    $nextUrl = $_SERVER['REQUEST_URI'];
                    $nextUrl = preg_replace('/([?&])page=\d+/', '', $nextUrl);
                    $nextUrl .= (strpos($nextUrl, '?') ? '&' : '?') . 'page=' . ($page + 1);
                    echo '<a href="' . htmlspecialchars($nextUrl) . '">Sonraki →</a>';
                }
                ?>
            </div>
        <?php endif; ?>

    <?php else: ?>
        <!-- No Products Found -->
        <div style="text-align: center; padding: 3rem;">
            <h3>Ürün Bulunamadı 😢</h3>
            <p>
                <?php 
                if ($search) {
                    echo "'" . htmlspecialchars($search) . "' için ürün bulunamadı.";
                } else {
                    echo "Bu kategoride şu an ürün yok.";
                }
                ?>
            </p>
            <a href="<?php echo APP_URL; ?>/products.php" class="btn btn-primary" style="margin-top: 1rem;">
                Tüm Ürünleri Gör
            </a>
        </div>
    <?php endif; ?>
</div>

<!-- JavaScript for Cart -->
<script>
function addToCart(productId, productTitle) {
    // Get cart from localStorage
    let cart = JSON.parse(localStorage.getItem('cart')) || [];

    // Check if product already in cart
    const existingItem = cart.find(item => item.id === productId);

    if (existingItem) {
        existingItem.quantity += 1;
    } else {
        cart.push({
            id: productId,
            title: productTitle,
            quantity: 1
        });
    }

    // Save to localStorage
    localStorage.setItem('cart', JSON.stringify(cart));

    // Update cart count
    updateCartCount();

    // Show notification
    alert(productTitle + ' sepete eklendi!');
}

function updateCartCount() {
    const cart = JSON.parse(localStorage.getItem('cart')) || [];
    const totalItems = cart.reduce((sum, item) => sum + item.quantity, 0);
    document.getElementById('cart-count').textContent = totalItems;
}

// Update cart count on page load
window.addEventListener('DOMContentLoaded', updateCartCount);
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
