<?php
/**
 * User Registration Page
 */

// Set page title
$pageTitle = 'Kayıt Ol';
$pageDescription = 'CDVectors hesabınızı oluşturun ve premium vektör grafiklerine erişin.';

require_once __DIR__ . '/includes/header.php';
require_once __DIR__ . '/includes/auth.php';

// If already logged in, redirect to home
if (isset($_SESSION['user_id'])) {
    header('Location: ' . APP_URL);
    exit();
}

$auth = new Auth();
$errors = [];
$success = false;

// Process registration form
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Verify CSRF token
    if (!isset($_POST['csrf_token']) || !Auth::verifyCsrfToken($_POST['csrf_token'])) {
        $errors[] = 'Güvenlik doğrulaması başarısız. Lütfen tekrar deneyin.';
    } else {
        $result = $auth->register($_POST);
        if ($result['success']) {
            $success = true;
            $_SESSION['flash_message'] = $result['message'];
            $_SESSION['flash_type'] = 'success';
            // Redirect to login
            header('Location: ' . APP_URL . '/login.php');
            exit();
        } else {
            $errors = (array)$result['message'];
            if (is_string($errors[0])) {
                $errors = [$result['message']];
            }
        }
    }
}

// Generate CSRF token for form
$csrfToken = Auth::generateCsrfToken();
?>

<div class="auth-container">
    <div class="auth-box register-box">
        <h1>Kayıt Ol</h1>
        <p class="auth-subtitle">Hesabınızı oluşturun ve başlayın</p>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-error" role="alert">
                <h3>Hatalar:</h3>
                <ul>
                    <?php foreach ($errors as $error): ?>
                        <li><?php echo htmlspecialchars($error); ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <form method="POST" class="auth-form" action="<?php echo APP_URL; ?>/register.php">
            <!-- CSRF Token -->
            <input type="hidden" name="csrf_token" value="<?php echo $csrfToken; ?>">

            <!-- Username -->
            <div class="form-group">
                <label for="username">Kullanıcı Adı *</label>
                <input 
                    type="text" 
                    id="username" 
                    name="username" 
                    placeholder="kullanıcı_adınız"
                    value="<?php echo isset($_POST['username']) ? htmlspecialchars($_POST['username']) : ''; ?>"
                    required
                    minlength="3"
                    maxlength="50"
                    pattern="[a-zA-Z0-9_-]+"
                    title="Sadece harf, rakam, alt çizgi ve tire kullanabilirsiniz"
                >
                <small>3-50 karakter, sadece harf, rakam, alt çizgi (-) ve tire (_)</small>
            </div>

            <!-- Email -->
            <div class="form-group">
                <label for="email">E-posta Adresiniz *</label>
                <input 
                    type="email" 
                    id="email" 
                    name="email" 
                    placeholder="ornek@example.com"
                    value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>"
                    required
                    autocomplete="email"
                >
            </div>

            <!-- First Name -->
            <div class="form-group">
                <label for="first_name">Adınız</label>
                <input 
                    type="text" 
                    id="first_name" 
                    name="first_name" 
                    placeholder="Adınız"
                    value="<?php echo isset($_POST['first_name']) ? htmlspecialchars($_POST['first_name']) : ''; ?>"
                    maxlength="50"
                >
            </div>

            <!-- Last Name -->
            <div class="form-group">
                <label for="last_name">Soyadınız</label>
                <input 
                    type="text" 
                    id="last_name" 
                    name="last_name" 
                    placeholder="Soyadınız"
                    value="<?php echo isset($_POST['last_name']) ? htmlspecialchars($_POST['last_name']) : ''; ?>"
                    maxlength="50"
                >
            </div>

            <!-- Password -->
            <div class="form-group">
                <label for="password">Şifre *</label>
                <input 
                    type="password" 
                    id="password" 
                    name="password" 
                    placeholder="••••••••"
                    required
                    minlength="8"
                    autocomplete="new-password"
                >
                <small>En az 8 karakter, büyük harf, küçük harf ve rakam içermelidir</small>
            </div>

            <!-- Confirm Password -->
            <div class="form-group">
                <label for="confirm_password">Şifreyi Onayla *</label>
                <input 
                    type="password" 
                    id="confirm_password" 
                    name="confirm_password" 
                    placeholder="••••••••"
                    required
                    minlength="8"
                    autocomplete="new-password"
                >
            </div>

            <!-- Terms Checkbox -->
            <div class="form-group checkbox">
                <input 
                    type="checkbox" 
                    id="agree_terms" 
                    name="agree_terms" 
                    required
                >
                <label for="agree_terms">
                    <a href="<?php echo APP_URL; ?>/terms.php" target="_blank">Kullanım Şartları</a>'nı ve 
                    <a href="<?php echo APP_URL; ?>/privacy.php" target="_blank">Gizlilik Politikası</a>'nı kabul ediyorum *
                </label>
            </div>

            <!-- Submit Button -->
            <button type="submit" class="btn btn-primary btn-block">
                Kayıt Ol
            </button>
        </form>

        <!-- Login Link -->
        <p class="auth-footer">
            Zaten hesabınız var mı? 
            <a href="<?php echo APP_URL; ?>/login.php">Giriş yapın</a>
        </p>
    </div>
</div>

<style>
.auth-container {
    display: flex;
    justify-content: center;
    align-items: center;
    min-height: calc(100vh - 200px);
    padding: 20px;
}

.auth-box {
    background: white;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0, 0, 0, 0.1);
    padding: 40px;
    max-width: 400px;
    width: 100%;
}

.auth-box h1 {
    margin: 0 0 10px 0;
    color: #333;
    font-size: 24px;
}

.auth-subtitle {
    color: #666;
    margin: 0 0 20px 0;
    font-size: 14px;
}

.auth-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-group label {
    margin-bottom: 5px;
    font-weight: 500;
    color: #333;
}

.form-group input {
    padding: 10px 12px;
    border: 1px solid #ddd;
    border-radius: 4px;
    font-size: 14px;
    transition: border-color 0.3s;
}

.form-group input:focus {
    outline: none;
    border-color: #007bff;
    box-shadow: 0 0 0 3px rgba(0, 123, 255, 0.1);
}

.form-group small {
    color: #999;
    font-size: 12px;
    margin-top: 3px;
}

.form-group.checkbox {
    flex-direction: row;
    align-items: flex-start;
    gap: 8px;
}

.form-group.checkbox input {
    width: auto;
    margin-top: 5px;
}

.form-group.checkbox label {
    margin: 0;
    font-weight: normal;
    font-size: 14px;
}

.form-group.checkbox a {
    color: #007bff;
    text-decoration: none;
}

.form-group.checkbox a:hover {
    text-decoration: underline;
}

.btn {
    padding: 10px 15px;
    border: none;
    border-radius: 4px;
    font-weight: 500;
    cursor: pointer;
    transition: background-color 0.3s;
}

.btn-primary {
    background-color: #007bff;
    color: white;
}

.btn-primary:hover {
    background-color: #0056b3;
}

.btn-block {
    width: 100%;
}

.auth-footer {
    text-align: center;
    margin-top: 20px;
    color: #666;
    font-size: 14px;
}

.auth-footer a {
    color: #007bff;
    text-decoration: none;
}

.auth-footer a:hover {
    text-decoration: underline;
}

.alert {
    padding: 15px;
    border-radius: 4px;
    margin-bottom: 20px;
}

.alert-error {
    background-color: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

.alert h3 {
    margin: 0 0 10px 0;
    font-size: 16px;
}

.alert ul {
    margin: 0;
    padding-left: 20px;
}

.alert li {
    margin-bottom: 5px;
}
</style>

<?php require_once __DIR__ . '/includes/footer.php'; ?>
