<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/utils.php';

requireLogin();

$db = Database::getInstance()->getConnection();

if (isset($_GET['logout'])) {
    session_destroy();
    redirect('/admin/login.php');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'delete_product') {
        $productId = intval($_POST['product_id']);
        $stmt = $db->prepare("UPDATE products SET active = 0 WHERE id = ?");
        $stmt->execute([$productId]);
        setFlash('Product deleted successfully');
        redirect('/admin/dashboard.php');
    }
    
    if ($action === 'add_coupon') {
        $code = strtoupper(sanitize($_POST['code']));
        $discount = floatval($_POST['discount_percent']);
        $maxUses = $_POST['max_uses'] ? intval($_POST['max_uses']) : null;
        
        $stmt = $db->prepare("
            INSERT INTO coupons (code, discount_percent, max_uses, active) 
            VALUES (?, ?, ?, 1)
        ");
        $stmt->execute([$code, $discount, $maxUses]);
        setFlash('Coupon created successfully');
        logMessage("Coupon created: $code");
        redirect('/admin/dashboard.php');
    }
}

$totalRevenue = $db->query("SELECT COALESCE(SUM(amount), 0) as total FROM orders WHERE status = 'completed'")->fetch()['total'];
$totalOrders = $db->query("SELECT COUNT(*) as total FROM orders WHERE status = 'completed'")->fetch()['total'];
$totalProducts = $db->query("SELECT COUNT(*) as total FROM products WHERE active = 1")->fetch()['total'];
$totalCoupons = $db->query("SELECT COUNT(*) as total FROM coupons WHERE active = 1")->fetch()['total'];

$products = $db->query("SELECT * FROM products ORDER BY created_at DESC LIMIT 50")->fetchAll();
$orders = $db->query("SELECT o.*, p.name as product_name FROM orders o LEFT JOIN products p ON o.product_id = p.id ORDER BY o.created_at DESC LIMIT 50")->fetchAll();
$coupons = $db->query("SELECT * FROM coupons ORDER BY created_at DESC")->fetchAll();

$flash = getFlash();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - <?= SITE_NAME ?></title>
    <link rel="stylesheet" href="/assets/css/style.css">
</head>
<body>
    <header>
        <div class="container">
            <h1><?= SITE_NAME ?> - Admin Panel</h1>
            <nav>
                <a href="/">Home</a>
                <a href="/admin/import.php">Import CSV</a>
                <a href="?logout=1">Logout</a>
            </nav>
        </div>
    </header>

    <main class="container">
        <?php if ($flash): ?>
            <div class="alert alert-<?= $flash['type'] ?>">
                <?= htmlspecialchars($flash['message']) ?>
            </div>
        <?php endif; ?>

        <div class="admin-stats">
            <div class="stat-card">
                <h3>Total Revenue</h3>
                <p class="stat-value"><?= formatPrice($totalRevenue) ?></p>
            </div>
            <div class="stat-card">
                <h3>Total Orders</h3>
                <p class="stat-value"><?= $totalOrders ?></p>
            </div>
            <div class="stat-card">
                <h3>Active Products</h3>
                <p class="stat-value"><?= $totalProducts ?></p>
            </div>
            <div class="stat-card">
                <h3>Active Coupons</h3>
                <p class="stat-value"><?= $totalCoupons ?></p>
            </div>
        </div>

        <section class="admin-section">
            <h2>Create Coupon</h2>
            <form method="POST" action="">
                <input type="hidden" name="action" value="add_coupon">
                <div style="display: grid; grid-template-columns: 1fr 1fr 1fr; gap: 1rem;">
                    <div class="form-group">
                        <label>Coupon Code:</label>
                        <input type="text" name="code" required placeholder="SAVE20">
                    </div>
                    <div class="form-group">
                        <label>Discount %:</label>
                        <input type="number" step="0.01" name="discount_percent" required placeholder="20">
                    </div>
                    <div class="form-group">
                        <label>Max Uses (optional):</label>
                        <input type="number" name="max_uses" placeholder="100">
                    </div>
                </div>
                <button type="submit" class="btn btn-success">Create Coupon</button>
            </form>
        </section>

        <section class="admin-section">
            <h2>Products</h2>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Category</th>
                            <th>Price</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($products as $product): ?>
                            <tr>
                                <td>#<?= $product['csv_id'] ?? $product['id'] ?></td>
                                <td><?= htmlspecialchars(substr($product['name'], 0, 50)) ?></td>
                                <td><span class="badge badge-primary"><?= htmlspecialchars($product['category']) ?></span></td>
                                <td><?= formatPrice($product['price']) ?></td>
                                <td>
                                    <?php if ($product['active']): ?>
                                        <span class="badge badge-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge badge-danger">Inactive</span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="/product.php?id=<?= $product['id'] ?>" class="btn" style="padding: 0.4rem 0.8rem; font-size: 0.85rem;">View</a>
                                    <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this product?')">
                                        <input type="hidden" name="action" value="delete_product">
                                        <input type="hidden" name="product_id" value="<?= $product['id'] ?>">
                                        <button type="submit" class="btn btn-danger" style="padding: 0.4rem 0.8rem; font-size: 0.85rem;">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </section>

        <section class="admin-section">
            <h2>Recent Orders</h2>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Order ID</th>
                            <th>Product</th>
                            <th>Customer</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Delivery</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($orders as $order): ?>
                            <tr>
                                <td><?= htmlspecialchars($order['order_id']) ?></td>
                                <td><?= htmlspecialchars($order['product_name'] ?? 'N/A') ?></td>
                                <td><?= htmlspecialchars($order['customer_email']) ?></td>
                                <td><?= formatPrice($order['amount']) ?></td>
                                <td>
                                    <?php if ($order['status'] === 'completed'): ?>
                                        <span class="badge badge-success">Completed</span>
                                    <?php else: ?>
                                        <span class="badge badge-warning"><?= htmlspecialchars($order['status']) ?></span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($order['delivery_sent']): ?>
                                        <span class="badge badge-success">Sent</span>
                                    <?php else: ?>
                                        <span class="badge badge-danger">Pending</span>
                                    <?php endif; ?>
                                </td>
                                <td><?= formatDate($order['created_at']) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </section>

        <section class="admin-section">
            <h2>Coupons</h2>
            <div class="table-responsive">
                <table>
                    <thead>
                        <tr>
                            <th>Code</th>
                            <th>Discount %</th>
                            <th>Uses</th>
                            <th>Max Uses</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($coupons as $coupon): ?>
                            <tr>
                                <td><strong><?= htmlspecialchars($coupon['code']) ?></strong></td>
                                <td><?= $coupon['discount_percent'] ?>%</td>
                                <td><?= $coupon['current_uses'] ?></td>
                                <td><?= $coupon['max_uses'] ?? '∞' ?></td>
                                <td>
                                    <?php if ($coupon['active']): ?>
                                        <span class="badge badge-success">Active</span>
                                    <?php else: ?>
                                        <span class="badge badge-danger">Inactive</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </section>
    </main>

    <footer>
        <div class="container">
            <p>&copy; <?= date('Y') ?> <?= SITE_NAME ?> Admin Panel. All rights reserved.</p>
        </div>
    </footer>
</body>
</html>
