<?php
require_once 'includes/config.php';
require_once 'includes/db.php';
require_once 'includes/utils.php';

$db = Database::getInstance()->getConnection();

$category = isset($_GET['category']) ? sanitize($_GET['category']) : '';
$search = isset($_GET['search']) ? sanitize($_GET['search']) : '';
$sort = isset($_GET['sort']) ? sanitize($_GET['sort']) : 'newest';
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$itemsPerPage = 12;

$where = ['active = 1'];
$params = [];

if ($category && $category !== 'all') {
    $where[] = 'category = ?';
    $params[] = $category;
}

if ($search) {
    $where[] = '(name LIKE ? OR description LIKE ? OR tags LIKE ?)';
    $searchTerm = "%$search%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$whereClause = implode(' AND ', $where);

$countStmt = $db->prepare("SELECT COUNT(*) as total FROM products WHERE $whereClause");
$countStmt->execute($params);
$totalItems = $countStmt->fetch()['total'];

$pagination = getPaginationData($page, $totalItems, $itemsPerPage);

$orderBy = match($sort) {
    'price_low' => 'price ASC',
    'price_high' => 'price DESC',
    'popular' => 'popular DESC, created_at DESC',
    default => 'created_at DESC'
};

$stmt = $db->prepare("
    SELECT * FROM products 
    WHERE $whereClause
    ORDER BY $orderBy
    LIMIT {$pagination['items_per_page']} OFFSET {$pagination['offset']}
");
$stmt->execute($params);
$products = $stmt->fetchAll();

$categoriesStmt = $db->query("SELECT DISTINCT category FROM products WHERE active = 1 AND category IS NOT NULL");
$categories = $categoriesStmt->fetchAll(PDO::FETCH_COLUMN);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= SITE_NAME ?> - Premium Stock Vectors & Graphics</title>
    <link rel="stylesheet" href="/assets/css/style.css">
</head>
<body>
    <header>
        <div class="container">
            <h1><?= SITE_NAME ?></h1>
            <nav>
                <a href="/">Home</a>
                <a href="/admin/login.php">Admin</a>
            </nav>
        </div>
    </header>

    <main class="container">
        <section class="hero">
            <h2>Premium Stock Vectors & Graphics</h2>
            <p>High-quality digital assets for your creative projects</p>
        </section>

        <section class="filters">
            <form method="GET" action="/" class="filter-group">
                <input type="text" name="search" placeholder="Search products..." value="<?= htmlspecialchars($search) ?>">
                
                <select name="category">
                    <option value="all">All Categories</option>
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?= htmlspecialchars($cat) ?>" <?= $category === $cat ? 'selected' : '' ?>>
                            <?= ucfirst(htmlspecialchars($cat)) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                
                <select name="sort">
                    <option value="newest" <?= $sort === 'newest' ? 'selected' : '' ?>>Newest First</option>
                    <option value="popular" <?= $sort === 'popular' ? 'selected' : '' ?>>Most Popular</option>
                    <option value="price_low" <?= $sort === 'price_low' ? 'selected' : '' ?>>Price: Low to High</option>
                    <option value="price_high" <?= $sort === 'price_high' ? 'selected' : '' ?>>Price: High to Low</option>
                </select>
                
                <button type="submit" class="btn">Filter</button>
            </form>
        </section>

        <?php if (empty($products)): ?>
            <div class="alert alert-info">
                <p>No products found. Try adjusting your filters.</p>
            </div>
        <?php else: ?>
            <div class="product-grid">
                <?php foreach ($products as $product): ?>
                    <div class="product-card">
                        <?php if ($product['image_url']): ?>
                            <img src="<?= htmlspecialchars($product['image_url']) ?>" 
                                 alt="<?= htmlspecialchars($product['name']) ?>" 
                                 class="product-image">
                        <?php else: ?>
                            <div class="product-image placeholder">
                                <?= strtoupper(substr($product['name'], 0, 1)) ?>
                            </div>
                        <?php endif; ?>
                        
                        <div class="product-info">
                            <?php if ($product['category']): ?>
                                <span class="product-category"><?= htmlspecialchars($product['category']) ?></span>
                            <?php endif; ?>
                            
                            <h3 class="product-title"><?= htmlspecialchars($product['name']) ?></h3>
                            
                            <p class="product-description">
                                <?= truncateText(htmlspecialchars($product['description']), 100) ?>
                            </p>
                            
                            <div class="product-footer">
                                <span class="product-price"><?= formatPrice($product['price']) ?></span>
                                <a href="/product.php?id=<?= $product['id'] ?>" class="btn">View Details</a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

            <?php if ($pagination['total_pages'] > 1): ?>
                <div class="pagination">
                    <?php for ($i = 1; $i <= $pagination['total_pages']; $i++): ?>
                        <?php
                        $params = $_GET;
                        $params['page'] = $i;
                        $queryString = http_build_query($params);
                        ?>
                        <?php if ($i === $page): ?>
                            <span class="active"><?= $i ?></span>
                        <?php else: ?>
                            <a href="/?<?= $queryString ?>"><?= $i ?></a>
                        <?php endif; ?>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </main>

    <footer>
        <div class="container">
            <p>&copy; <?= date('Y') ?> <?= SITE_NAME ?>. All rights reserved.</p>
        </div>
    </footer>
</body>
</html>
