<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/utils.php';
require_once '../includes/mailer.php';

function verifyPaddleWebhook($data) {
    $signature = $data['p_signature'] ?? '';
    unset($data['p_signature']);
    
    ksort($data);
    $queryString = '';
    foreach ($data as $key => $value) {
        if (is_array($value)) {
            $value = $value[0];
        }
        $queryString .= "$key=$value";
    }
    
    $expectedSignature = hash_hmac('sha1', $queryString, PADDLE_SHARED_SECRET);
    
    return hash_equals($signature, $expectedSignature);
}

logMessage("Paddle webhook received: " . json_encode($_POST));

if (!verifyPaddleWebhook($_POST)) {
    logMessage("Webhook verification failed", 'ERROR');
    http_response_code(403);
    echo json_encode(['error' => 'Invalid signature']);
    exit;
}

$alertName = $_POST['alert_name'] ?? '';

if ($alertName === 'payment_succeeded') {
    try {
        $db = Database::getInstance()->getConnection();
        
        $productId = intval($_POST['product_id'] ?? 0);
        $customerEmail = $_POST['email'] ?? '';
        $customerName = $_POST['customer_name'] ?? '';
        $amount = floatval($_POST['sale_gross'] ?? 0);
        $currency = $_POST['currency'] ?? 'USD';
        $paddleOrderId = $_POST['order_id'] ?? '';
        
        $stmt = $db->prepare("SELECT * FROM products WHERE id = ?");
        $stmt->execute([$productId]);
        $product = $stmt->fetch();
        
        if (!$product) {
            logMessage("Product not found: $productId", 'ERROR');
            http_response_code(404);
            exit;
        }
        
        $orderId = generateOrderId();
        $downloadPassword = $orderId;
        
        $stmt = $db->prepare("
            INSERT INTO orders 
            (order_id, product_id, customer_email, customer_name, amount, currency, 
             paddle_order_id, status, download_password, delivery_sent)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'completed', ?, 0)
        ");
        
        $stmt->execute([
            $orderId,
            $productId,
            $customerEmail,
            $customerName,
            $amount,
            $currency,
            $paddleOrderId,
            $downloadPassword
        ]);
        
        $order = [
            'order_id' => $orderId,
            'customer_email' => $customerEmail,
            'customer_name' => $customerName,
            'amount' => $amount,
            'download_password' => $downloadPassword
        ];
        
        $downloadLinks = [];
        if (GITHUB_TOKEN && GITHUB_REPO && $product['github_release_tag']) {
            $githubUrl = "https://api.github.com/repos/" . GITHUB_REPO . "/releases/tags/" . $product['github_release_tag'];
            
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $githubUrl);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, [
                'User-Agent: VectorShop',
                'Authorization: token ' . GITHUB_TOKEN
            ]);
            
            $response = curl_exec($ch);
            curl_close($ch);
            
            if ($response) {
                $release = json_decode($response, true);
                if (isset($release['assets'])) {
                    foreach ($release['assets'] as $asset) {
                        $downloadLinks[] = [
                            'name' => $asset['name'],
                            'url' => $asset['browser_download_url'],
                            'size' => $asset['size']
                        ];
                    }
                }
            }
        }
        
        if (empty($downloadLinks)) {
            $downloadLinks[] = [
                'name' => $product['name'] . '.zip',
                'url' => SITE_URL . '/downloads/' . $product['id'],
                'size' => 1024 * 1024
            ];
        }
        
        $emailSent = sendOrderEmail($order, $product, $downloadLinks);
        
        if ($emailSent) {
            $db->prepare("UPDATE orders SET delivery_sent = 1 WHERE order_id = ?")
               ->execute([$orderId]);
        }
        
        logMessage("Order completed: $orderId - {$product['name']} - $customerEmail");
        
        echo json_encode(['status' => 'success', 'order_id' => $orderId]);
        
    } catch (Exception $e) {
        logMessage("Webhook processing error: " . $e->getMessage(), 'ERROR');
        http_response_code(500);
        echo json_encode(['error' => $e->getMessage()]);
    }
} else {
    logMessage("Unknown webhook event: $alertName", 'WARNING');
    echo json_encode(['status' => 'ignored']);
}
?>
